// Copyright 2025 Specter Ops, Inc.
//
// Licensed under the Apache License, Version 2.0
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//	http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0
package generator

import (
	"fmt"
	"os"
	"path/filepath"

	"github.com/specterops/bloodhound/packages/go/schemagen/csgen"
	"github.com/specterops/bloodhound/packages/go/schemagen/model"
)

type Schema struct {
	Common          model.Graph
	ActiveDirectory model.ActiveDirectory
}

func GenerateCSharpBindings(projectRoot string, commonSchema model.Graph, adSchema model.ActiveDirectory) error {
	path := filepath.Join(projectRoot, "packages/csharp/graphschema/PropertyNames.cs")
	nodes := make([]csgen.SyntaxNode, 0)
	for _, property := range commonSchema.Properties {
		nodes = append(nodes, csgen.BinaryExpression{
			LeftOperand: csgen.ClassMemberAssignment{
				Visibility: csgen.VisibilityPublic,
				Modifier:   csgen.Modifiers{csgen.ModifierStatic, csgen.ModifierReadonly},
				Type:       csgen.TypeString,
				Symbol:     csgen.Symbol(property.Symbol),
			},
			Operator: csgen.OperatorEquals,
			RightOperand: csgen.Literal{
				Value: property.Representation,
				Null:  false,
			},
		})
	}

	for _, property := range adSchema.Properties {
		nodes = append(nodes, csgen.BinaryExpression{
			LeftOperand: csgen.ClassMemberAssignment{
				Visibility: csgen.VisibilityPublic,
				Modifier:   csgen.Modifiers{csgen.ModifierStatic, csgen.ModifierReadonly},
				Type:       csgen.TypeString,
				Symbol:     csgen.Symbol(property.Symbol),
			},
			Operator: csgen.OperatorEquals,
			RightOperand: csgen.Literal{
				Value: property.Representation,
				Null:  false,
			},
		})
	}
	topLevel := csgen.Namespace{
		Name: "SharpHoundCommon.Enums",
		Nodes: []csgen.SyntaxNode{
			csgen.Class{
				Modifiers:  csgen.Modifiers{csgen.ModifierStatic},
				Visibility: csgen.VisibilityPublic,
				Name:       "PropertyNames",
				Nodes:      nodes,
				Annotations: []csgen.Annotation{
					{
						HasContent:        true,
						AnnotationType:    "GeneratedCode",
						AnnotationContent: []string{"cuelang-codegen", "v1.0.0"},
					},
				},
			},
		},
	}

	builder := csgen.NewOutputBuilder()
	builder.Write(fmt.Sprintf("// <auto-generated/>\n// Code generated by Cuelang code gen. DO NOT EDIT!\n// Cuelang source: %s/\n", SchemaSourceName))
	if err := csgen.WalkSyntaxTree(topLevel, builder); err != nil {
		return err
	} else if err := os.WriteFile(path, []byte(builder.Build()), 0644); err != nil {
		return err
	}

	return nil
}
