//! For arithmetic operations which cannot trigger Illegal Behavior, this test evaluates those
//! operations with undefined operands (or partially-undefined vector operands), and then uses
//! `@compileLog` to validate the results are as expected in terms of comptime-known undefined.

export fn entry() void {
    testInt(u8);
    testInt(i8);
    testInt(u32);
    testInt(i32);
    testInt(u500);
    testInt(i500);

    testFloat(f16);
    testFloat(f32);
    testFloat(f64);
    testFloat(f80);
    testFloat(f128);
}

fn testInt(comptime Int: type) void {
    testIntWithValue(Int, 3); // all comptime-known
    testIntWithValue(Int, struct {
        var rt: Int = 0;
    }.rt); // partially runtime-known
}

fn testFloat(comptime Float: type) void {
    testFloatWithValue(Float, 3.0); // all comptime-known
    testFloatWithValue(Float, struct {
        var rt: Float = 0.0;
    }.rt); // partially runtime-known
}

inline fn testIntWithValue(comptime Int: type, x: Int) void {
    const V = @Vector(2, Int);
    const u: Int = undefined;

    // Wrapping addition

    @compileLog(x +% u); // undef
    @compileLog(u +% x); // undef

    @compileLog(V{ x, u } +% V{ x, x }); // { 6, undef }
    @compileLog(V{ u, x } +% V{ x, x }); // { undef, 6 }
    @compileLog(V{ u, u } +% V{ x, x }); // undef

    @compileLog(V{ x, x } +% V{ x, u }); // { 6, undef }
    @compileLog(V{ x, u } +% V{ x, u }); // { 6, undef }
    @compileLog(V{ u, x } +% V{ x, u }); // undef
    @compileLog(V{ u, u } +% V{ x, u }); // undef

    @compileLog(V{ x, x } +% V{ u, x }); // { undef, 6 }
    @compileLog(V{ x, u } +% V{ u, x }); // undef
    @compileLog(V{ u, x } +% V{ u, x }); // { undef, 6 }
    @compileLog(V{ u, u } +% V{ u, x }); // undef

    @compileLog(V{ x, x } +% V{ u, u }); // undef
    @compileLog(V{ x, u } +% V{ u, u }); // undef
    @compileLog(V{ u, x } +% V{ u, u }); // undef
    @compileLog(V{ u, u } +% V{ u, u }); // undef

    // Saturating addition

    @compileLog(x +| u); // undef
    @compileLog(u +| x); // undef

    @compileLog(V{ x, u } +| V{ x, x }); // { 6, undef }
    @compileLog(V{ u, x } +| V{ x, x }); // { undef, 6 }
    @compileLog(V{ u, u } +| V{ x, x }); // undef

    @compileLog(V{ x, x } +| V{ x, u }); // { 6, undef }
    @compileLog(V{ x, u } +| V{ x, u }); // { 6, undef }
    @compileLog(V{ u, x } +| V{ x, u }); // undef
    @compileLog(V{ u, u } +| V{ x, u }); // undef

    @compileLog(V{ x, x } +| V{ u, x }); // { undef, 6 }
    @compileLog(V{ x, u } +| V{ u, x }); // undef
    @compileLog(V{ u, x } +| V{ u, x }); // { undef, 6 }
    @compileLog(V{ u, u } +| V{ u, x }); // undef

    @compileLog(V{ x, x } +| V{ u, u }); // undef
    @compileLog(V{ x, u } +| V{ u, u }); // undef
    @compileLog(V{ u, x } +| V{ u, u }); // undef
    @compileLog(V{ u, u } +| V{ u, u }); // undef

    // Wrapping subtraction

    @compileLog(x -% u); // undef
    @compileLog(u -% x); // undef

    @compileLog(V{ x, u } -% V{ x, x }); // { 0, undef }
    @compileLog(V{ u, x } -% V{ x, x }); // { undef, 0 }
    @compileLog(V{ u, u } -% V{ x, x }); // undef

    @compileLog(V{ x, x } -% V{ x, u }); // { 0, undef }
    @compileLog(V{ x, u } -% V{ x, u }); // { 0, undef }
    @compileLog(V{ u, x } -% V{ x, u }); // undef
    @compileLog(V{ u, u } -% V{ x, u }); // undef

    @compileLog(V{ x, x } -% V{ u, x }); // { undef, 0 }
    @compileLog(V{ x, u } -% V{ u, x }); // undef
    @compileLog(V{ u, x } -% V{ u, x }); // { undef, 0 }
    @compileLog(V{ u, u } -% V{ u, x }); // undef

    @compileLog(V{ x, x } -% V{ u, u }); // undef
    @compileLog(V{ x, u } -% V{ u, u }); // undef
    @compileLog(V{ u, x } -% V{ u, u }); // undef
    @compileLog(V{ u, u } -% V{ u, u }); // undef

    // Saturating subtraction

    @compileLog(x -| u); // undef
    @compileLog(u -| x); // undef

    @compileLog(V{ x, u } -| V{ x, x }); // { 0, undef }
    @compileLog(V{ u, x } -| V{ x, x }); // { undef, 0 }
    @compileLog(V{ u, u } -| V{ x, x }); // undef

    @compileLog(V{ x, x } -| V{ x, u }); // { 0, undef }
    @compileLog(V{ x, u } -| V{ x, u }); // { 0, undef }
    @compileLog(V{ u, x } -| V{ x, u }); // undef
    @compileLog(V{ u, u } -| V{ x, u }); // undef

    @compileLog(V{ x, x } -| V{ u, x }); // { undef, 0 }
    @compileLog(V{ x, u } -| V{ u, x }); // undef
    @compileLog(V{ u, x } -| V{ u, x }); // { undef, 0 }
    @compileLog(V{ u, u } -| V{ u, x }); // undef

    @compileLog(V{ x, x } -| V{ u, u }); // undef
    @compileLog(V{ x, u } -| V{ u, u }); // undef
    @compileLog(V{ u, x } -| V{ u, u }); // undef
    @compileLog(V{ u, u } -| V{ u, u }); // undef

    // Wrapping multiplication

    @compileLog(x *% u); // undef
    @compileLog(u *% x); // undef

    @compileLog(V{ x, u } *% V{ x, x }); // { 9, undef }
    @compileLog(V{ u, x } *% V{ x, x }); // { undef, 9 }
    @compileLog(V{ u, u } *% V{ x, x }); // undef

    @compileLog(V{ x, x } *% V{ x, u }); // { 9, undef }
    @compileLog(V{ x, u } *% V{ x, u }); // { 9, undef }
    @compileLog(V{ u, x } *% V{ x, u }); // undef
    @compileLog(V{ u, u } *% V{ x, u }); // undef

    @compileLog(V{ x, x } *% V{ u, x }); // { undef, 9 }
    @compileLog(V{ x, u } *% V{ u, x }); // undef
    @compileLog(V{ u, x } *% V{ u, x }); // { undef, 9 }
    @compileLog(V{ u, u } *% V{ u, x }); // undef

    @compileLog(V{ x, x } *% V{ u, u }); // undef
    @compileLog(V{ x, u } *% V{ u, u }); // undef
    @compileLog(V{ u, x } *% V{ u, u }); // undef
    @compileLog(V{ u, u } *% V{ u, u }); // undef

    // Saturating multiplication

    @compileLog(x *| u); // undef
    @compileLog(u *| x); // undef

    @compileLog(V{ x, u } *| V{ x, x }); // { 9, undef }
    @compileLog(V{ u, x } *| V{ x, x }); // { undef, 9 }
    @compileLog(V{ u, u } *| V{ x, x }); // undef

    @compileLog(V{ x, x } *| V{ x, u }); // { 9, undef }
    @compileLog(V{ x, u } *| V{ x, u }); // { 9, undef }
    @compileLog(V{ u, x } *| V{ x, u }); // undef
    @compileLog(V{ u, u } *| V{ x, u }); // undef

    @compileLog(V{ x, x } *| V{ u, x }); // { undef, 9 }
    @compileLog(V{ x, u } *| V{ u, x }); // undef
    @compileLog(V{ u, x } *| V{ u, x }); // { undef, 9 }
    @compileLog(V{ u, u } *| V{ u, x }); // undef

    @compileLog(V{ x, x } *| V{ u, u }); // undef
    @compileLog(V{ x, u } *| V{ u, u }); // undef
    @compileLog(V{ u, x } *| V{ u, u }); // undef
    @compileLog(V{ u, u } *| V{ u, u }); // undef

    // Saturating shift

    if (@typeInfo(Int).int.signedness == .unsigned) {
        @compileLog(x <<| u); // undef
        @compileLog(u <<| x); // undef

        @compileLog(V{ x, u } <<| V{ x, x }); // { 24, undef }
        @compileLog(V{ u, x } <<| V{ x, x }); // { undef, 24 }
        @compileLog(V{ u, u } <<| V{ x, x }); // undef

        @compileLog(V{ x, x } <<| V{ x, u }); // { 24, undef }
        @compileLog(V{ x, u } <<| V{ x, u }); // { 24, undef }
        @compileLog(V{ u, x } <<| V{ x, u }); // undef
        @compileLog(V{ u, u } <<| V{ x, u }); // undef

        @compileLog(V{ x, x } <<| V{ u, x }); // { undef, 24 }
        @compileLog(V{ x, u } <<| V{ u, x }); // undef
        @compileLog(V{ u, x } <<| V{ u, x }); // { undef, 24 }
        @compileLog(V{ u, u } <<| V{ u, x }); // undef

        @compileLog(V{ x, x } <<| V{ u, u }); // undef
        @compileLog(V{ x, u } <<| V{ u, u }); // undef
        @compileLog(V{ u, x } <<| V{ u, u }); // undef
        @compileLog(V{ u, u } <<| V{ u, u }); // undef
    }

    // Bitwise XOR

    @compileLog(x ^ u); // undef
    @compileLog(u ^ x); // undef

    @compileLog(V{ x, u } ^ V{ x, x }); // { 0, undef }
    @compileLog(V{ u, x } ^ V{ x, x }); // { undef, 0 }
    @compileLog(V{ u, u } ^ V{ x, x }); // undef

    @compileLog(V{ x, x } ^ V{ x, u }); // { 0, undef }
    @compileLog(V{ x, u } ^ V{ x, u }); // { 0, undef }
    @compileLog(V{ u, x } ^ V{ x, u }); // undef
    @compileLog(V{ u, u } ^ V{ x, u }); // undef

    @compileLog(V{ x, x } ^ V{ u, x }); // { undef, 0 }
    @compileLog(V{ x, u } ^ V{ u, x }); // undef
    @compileLog(V{ u, x } ^ V{ u, x }); // { undef, 0 }
    @compileLog(V{ u, u } ^ V{ u, x }); // undef

    @compileLog(V{ x, x } ^ V{ u, u }); // undef
    @compileLog(V{ x, u } ^ V{ u, u }); // undef
    @compileLog(V{ u, x } ^ V{ u, u }); // undef
    @compileLog(V{ u, u } ^ V{ u, u }); // undef

    // Bitwise NOT

    @compileLog(~u); // undef
    @compileLog(~V{ u, u }); // undef

    if (Int == u8) { // Result depends on integer type
        @compileLog(~V{ x, u }); // { 252, undef }
        @compileLog(~V{ u, x }); // { undef, 252 }
    }

    // Other binary bitwise operations

    @compileLog(u & u); // undef
    @compileLog(u | u); // undef

    @compileLog(V{ u, u } & V{ u, u }); // undef
    @compileLog(V{ u, u } | V{ u, u }); // undef

    // Truncate

    if (@typeInfo(Int).int.signedness == .unsigned) {
        const W = @Vector(2, u1);
        @compileLog(@as(u1, @truncate(u))); // undef
        @compileLog(@as(W, @truncate(V{ x, u }))); // { 1, undef }
        @compileLog(@as(W, @truncate(V{ u, x }))); // { undef, 1 }
        @compileLog(@as(W, @truncate(V{ u, u }))); // undef
    }

    // Bit reverse

    @compileLog(@bitReverse(u)); // undef
    @compileLog(@bitReverse(V{ u, u })); // undef

    if (Int == u8) { // Result depends on integer type
        @compileLog(@bitReverse(V{ x, u })); // { 192, undef }
        @compileLog(@bitReverse(V{ u, x })); // { undef, 192 }
    }

    // Byte swap

    if (Int == u8) { // Result depends on integer type and is illegal for some
        @compileLog(@byteSwap(u)); // undef
        @compileLog(@byteSwap(V{ u, u })); // undef

        @compileLog(@byteSwap(V{ x, u })); // { 3, undef }
        @compileLog(@byteSwap(V{ u, x })); // { undef, 3 }
    }
}

inline fn testFloatWithValue(comptime Float: type, x: Float) void {
    const V = @Vector(2, Float);
    const u: Float = undefined;

    // Addition

    @compileLog(x + u); // undef
    @compileLog(u + x); // undef

    @compileLog(V{ x, u } + V{ x, x }); // { 6, undef }
    @compileLog(V{ u, x } + V{ x, x }); // { undef, 6 }
    @compileLog(V{ u, u } + V{ x, x }); // undef

    @compileLog(V{ x, x } + V{ x, u }); // { 6, undef }
    @compileLog(V{ x, u } + V{ x, u }); // { 6, undef }
    @compileLog(V{ u, x } + V{ x, u }); // undef
    @compileLog(V{ u, u } + V{ x, u }); // undef

    @compileLog(V{ x, x } + V{ u, x }); // { undef, 6 }
    @compileLog(V{ x, u } + V{ u, x }); // undef
    @compileLog(V{ u, x } + V{ u, x }); // { undef, 6 }
    @compileLog(V{ u, u } + V{ u, x }); // undef

    @compileLog(V{ x, x } + V{ u, u }); // undef
    @compileLog(V{ x, u } + V{ u, u }); // undef
    @compileLog(V{ u, x } + V{ u, u }); // undef
    @compileLog(V{ u, u } + V{ u, u }); // undef

    // Subtraction

    @compileLog(x - u); // undef
    @compileLog(u - x); // undef

    @compileLog(V{ x, u } - V{ x, x }); // { 0, undef }
    @compileLog(V{ u, x } - V{ x, x }); // { undef, 0 }
    @compileLog(V{ u, u } - V{ x, x }); // undef

    @compileLog(V{ x, x } - V{ x, u }); // { 0, undef }
    @compileLog(V{ x, u } - V{ x, u }); // { 0, undef }
    @compileLog(V{ u, x } - V{ x, u }); // undef
    @compileLog(V{ u, u } - V{ x, u }); // undef

    @compileLog(V{ x, x } - V{ u, x }); // { undef, 0 }
    @compileLog(V{ x, u } - V{ u, x }); // undef
    @compileLog(V{ u, x } - V{ u, x }); // { undef, 0 }
    @compileLog(V{ u, u } - V{ u, x }); // undef

    @compileLog(V{ x, x } - V{ u, u }); // undef
    @compileLog(V{ x, u } - V{ u, u }); // undef
    @compileLog(V{ u, x } - V{ u, u }); // undef
    @compileLog(V{ u, u } - V{ u, u }); // undef

    // Multiplication

    @compileLog(x * u); // undef
    @compileLog(u * x); // undef

    @compileLog(V{ x, u } * V{ x, x }); // { 9, undef }
    @compileLog(V{ u, x } * V{ x, x }); // { undef, 9 }
    @compileLog(V{ u, u } * V{ x, x }); // undef

    @compileLog(V{ x, x } * V{ x, u }); // { 9, undef }
    @compileLog(V{ x, u } * V{ x, u }); // { 9, undef }
    @compileLog(V{ u, x } * V{ x, u }); // undef
    @compileLog(V{ u, u } * V{ x, u }); // undef

    @compileLog(V{ x, x } * V{ u, x }); // { undef, 9 }
    @compileLog(V{ x, u } * V{ u, x }); // undef
    @compileLog(V{ u, x } * V{ u, x }); // { undef, 9 }
    @compileLog(V{ u, u } * V{ u, x }); // undef

    @compileLog(V{ x, x } * V{ u, u }); // undef
    @compileLog(V{ x, u } * V{ u, u }); // undef
    @compileLog(V{ u, x } * V{ u, u }); // undef
    @compileLog(V{ u, u } * V{ u, u }); // undef

    // Negation

    @compileLog(-u); // undef
    @compileLog(-V{ x, u }); // { -3, undef }
    @compileLog(-V{ u, x }); // { undef, -3 }
    @compileLog(-V{ u, u }); // undef
}

// error
//
// :40:5: error: found compile log statement
// :40:5: note: also here (5 times)
// :287:5: note: also here (5 times)
//
// Compile Log Output:
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), .{ 6, undefined })
// @as(@Vector(2, u8), .{ undefined, 6 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 6, undefined })
// @as(@Vector(2, u8), .{ 6, undefined })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 6 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 6 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), .{ 6, undefined })
// @as(@Vector(2, u8), .{ undefined, 6 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 6, undefined })
// @as(@Vector(2, u8), .{ 6, undefined })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 6 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 6 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), .{ 9, undefined })
// @as(@Vector(2, u8), .{ undefined, 9 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 9, undefined })
// @as(@Vector(2, u8), .{ 9, undefined })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 9 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 9 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), .{ 9, undefined })
// @as(@Vector(2, u8), .{ undefined, 9 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 9, undefined })
// @as(@Vector(2, u8), .{ 9, undefined })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 9 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 9 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), .{ 24, undefined })
// @as(@Vector(2, u8), .{ undefined, 24 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 24, undefined })
// @as(@Vector(2, u8), .{ 24, undefined })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 24 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 24 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), .{ 0, undefined })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ undefined, 0 })
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 252, undefined })
// @as(@Vector(2, u8), .{ undefined, 252 })
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u1, undefined)
// @as(@Vector(2, u1), .{ 1, undefined })
// @as(@Vector(2, u1), .{ undefined, 1 })
// @as(@Vector(2, u1), undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 192, undefined })
// @as(@Vector(2, u8), .{ undefined, 192 })
// @as(u8, undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), .{ 3, undefined })
// @as(@Vector(2, u8), .{ undefined, 3 })
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u8, [runtime value])
// @as(u8, [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(u8, undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), undefined)
// @as(u1, undefined)
// @as(@Vector(2, u1), [runtime value])
// @as(@Vector(2, u1), [runtime value])
// @as(@Vector(2, u1), undefined)
// @as(u8, undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(u8, undefined)
// @as(@Vector(2, u8), undefined)
// @as(@Vector(2, u8), [runtime value])
// @as(@Vector(2, u8), [runtime value])
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), .{ 6, undefined })
// @as(@Vector(2, i8), .{ undefined, 6 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ 6, undefined })
// @as(@Vector(2, i8), .{ 6, undefined })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 6 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 6 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), .{ 6, undefined })
// @as(@Vector(2, i8), .{ undefined, 6 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ 6, undefined })
// @as(@Vector(2, i8), .{ 6, undefined })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 6 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 6 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), .{ 9, undefined })
// @as(@Vector(2, i8), .{ undefined, 9 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ 9, undefined })
// @as(@Vector(2, i8), .{ 9, undefined })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 9 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 9 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), .{ 9, undefined })
// @as(@Vector(2, i8), .{ undefined, 9 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ 9, undefined })
// @as(@Vector(2, i8), .{ 9, undefined })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 9 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 9 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), .{ 0, undefined })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), .{ undefined, 0 })
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, [runtime value])
// @as(i8, [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), [runtime value])
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), undefined)
// @as(@Vector(2, i8), undefined)
// @as(i8, undefined)
// @as(@Vector(2, i8), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), .{ 6, undefined })
// @as(@Vector(2, u32), .{ undefined, 6 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ 6, undefined })
// @as(@Vector(2, u32), .{ 6, undefined })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 6 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 6 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), .{ 6, undefined })
// @as(@Vector(2, u32), .{ undefined, 6 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ 6, undefined })
// @as(@Vector(2, u32), .{ 6, undefined })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 6 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 6 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), .{ 9, undefined })
// @as(@Vector(2, u32), .{ undefined, 9 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ 9, undefined })
// @as(@Vector(2, u32), .{ 9, undefined })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 9 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 9 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), .{ 9, undefined })
// @as(@Vector(2, u32), .{ undefined, 9 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ 9, undefined })
// @as(@Vector(2, u32), .{ 9, undefined })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 9 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 9 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), .{ 24, undefined })
// @as(@Vector(2, u32), .{ undefined, 24 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ 24, undefined })
// @as(@Vector(2, u32), .{ 24, undefined })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 24 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 24 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), .{ 0, undefined })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), .{ undefined, 0 })
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u1, undefined)
// @as(@Vector(2, u1), .{ 1, undefined })
// @as(@Vector(2, u1), .{ undefined, 1 })
// @as(@Vector(2, u1), undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, [runtime value])
// @as(u32, [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), [runtime value])
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), undefined)
// @as(u32, undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), undefined)
// @as(@Vector(2, u32), undefined)
// @as(u1, undefined)
// @as(@Vector(2, u1), [runtime value])
// @as(@Vector(2, u1), [runtime value])
// @as(@Vector(2, u1), undefined)
// @as(u32, undefined)
// @as(@Vector(2, u32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), .{ 6, undefined })
// @as(@Vector(2, i32), .{ undefined, 6 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ 6, undefined })
// @as(@Vector(2, i32), .{ 6, undefined })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 6 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 6 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), .{ 6, undefined })
// @as(@Vector(2, i32), .{ undefined, 6 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ 6, undefined })
// @as(@Vector(2, i32), .{ 6, undefined })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 6 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 6 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), .{ 9, undefined })
// @as(@Vector(2, i32), .{ undefined, 9 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ 9, undefined })
// @as(@Vector(2, i32), .{ 9, undefined })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 9 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 9 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), .{ 9, undefined })
// @as(@Vector(2, i32), .{ undefined, 9 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ 9, undefined })
// @as(@Vector(2, i32), .{ 9, undefined })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 9 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 9 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), .{ 0, undefined })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), .{ undefined, 0 })
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, [runtime value])
// @as(i32, [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), [runtime value])
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), undefined)
// @as(@Vector(2, i32), undefined)
// @as(i32, undefined)
// @as(@Vector(2, i32), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), .{ 6, undefined })
// @as(@Vector(2, u500), .{ undefined, 6 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ 6, undefined })
// @as(@Vector(2, u500), .{ 6, undefined })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 6 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 6 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), .{ 6, undefined })
// @as(@Vector(2, u500), .{ undefined, 6 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ 6, undefined })
// @as(@Vector(2, u500), .{ 6, undefined })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 6 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 6 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), .{ 9, undefined })
// @as(@Vector(2, u500), .{ undefined, 9 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ 9, undefined })
// @as(@Vector(2, u500), .{ 9, undefined })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 9 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 9 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), .{ 9, undefined })
// @as(@Vector(2, u500), .{ undefined, 9 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ 9, undefined })
// @as(@Vector(2, u500), .{ 9, undefined })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 9 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 9 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), .{ 24, undefined })
// @as(@Vector(2, u500), .{ undefined, 24 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ 24, undefined })
// @as(@Vector(2, u500), .{ 24, undefined })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 24 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 24 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), .{ 0, undefined })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), .{ undefined, 0 })
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u1, undefined)
// @as(@Vector(2, u1), .{ 1, undefined })
// @as(@Vector(2, u1), .{ undefined, 1 })
// @as(@Vector(2, u1), undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, [runtime value])
// @as(u500, [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), [runtime value])
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), undefined)
// @as(u500, undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), undefined)
// @as(@Vector(2, u500), undefined)
// @as(u1, undefined)
// @as(@Vector(2, u1), [runtime value])
// @as(@Vector(2, u1), [runtime value])
// @as(@Vector(2, u1), undefined)
// @as(u500, undefined)
// @as(@Vector(2, u500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), .{ 6, undefined })
// @as(@Vector(2, i500), .{ undefined, 6 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ 6, undefined })
// @as(@Vector(2, i500), .{ 6, undefined })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 6 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 6 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), .{ 6, undefined })
// @as(@Vector(2, i500), .{ undefined, 6 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ 6, undefined })
// @as(@Vector(2, i500), .{ 6, undefined })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 6 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 6 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), .{ 9, undefined })
// @as(@Vector(2, i500), .{ undefined, 9 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ 9, undefined })
// @as(@Vector(2, i500), .{ 9, undefined })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 9 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 9 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), .{ 9, undefined })
// @as(@Vector(2, i500), .{ undefined, 9 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ 9, undefined })
// @as(@Vector(2, i500), .{ 9, undefined })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 9 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 9 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), .{ 0, undefined })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), .{ undefined, 0 })
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, [runtime value])
// @as(i500, [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), [runtime value])
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), undefined)
// @as(@Vector(2, i500), undefined)
// @as(i500, undefined)
// @as(@Vector(2, i500), undefined)
// @as(f16, undefined)
// @as(f16, undefined)
// @as(@Vector(2, f16), .{ 6, undefined })
// @as(@Vector(2, f16), .{ undefined, 6 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ 6, undefined })
// @as(@Vector(2, f16), .{ 6, undefined })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ undefined, 6 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ undefined, 6 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(f16, undefined)
// @as(f16, undefined)
// @as(@Vector(2, f16), .{ 0, undefined })
// @as(@Vector(2, f16), .{ undefined, 0 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ 0, undefined })
// @as(@Vector(2, f16), .{ 0, undefined })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ undefined, 0 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ undefined, 0 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(f16, undefined)
// @as(f16, undefined)
// @as(@Vector(2, f16), .{ 9, undefined })
// @as(@Vector(2, f16), .{ undefined, 9 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ 9, undefined })
// @as(@Vector(2, f16), .{ 9, undefined })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ undefined, 9 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), .{ undefined, 9 })
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(f16, undefined)
// @as(@Vector(2, f16), .{ -3, undefined })
// @as(@Vector(2, f16), .{ undefined, -3 })
// @as(@Vector(2, f16), undefined)
// @as(f16, undefined)
// @as(f16, undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(f16, undefined)
// @as(f16, undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(f16, undefined)
// @as(f16, undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(@Vector(2, f16), undefined)
// @as(f16, undefined)
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), [runtime value])
// @as(@Vector(2, f16), undefined)
// @as(f32, undefined)
// @as(f32, undefined)
// @as(@Vector(2, f32), .{ 6, undefined })
// @as(@Vector(2, f32), .{ undefined, 6 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ 6, undefined })
// @as(@Vector(2, f32), .{ 6, undefined })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ undefined, 6 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ undefined, 6 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(f32, undefined)
// @as(f32, undefined)
// @as(@Vector(2, f32), .{ 0, undefined })
// @as(@Vector(2, f32), .{ undefined, 0 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ 0, undefined })
// @as(@Vector(2, f32), .{ 0, undefined })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ undefined, 0 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ undefined, 0 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(f32, undefined)
// @as(f32, undefined)
// @as(@Vector(2, f32), .{ 9, undefined })
// @as(@Vector(2, f32), .{ undefined, 9 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ 9, undefined })
// @as(@Vector(2, f32), .{ 9, undefined })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ undefined, 9 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), .{ undefined, 9 })
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(f32, undefined)
// @as(@Vector(2, f32), .{ -3, undefined })
// @as(@Vector(2, f32), .{ undefined, -3 })
// @as(@Vector(2, f32), undefined)
// @as(f32, undefined)
// @as(f32, undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(f32, undefined)
// @as(f32, undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(f32, undefined)
// @as(f32, undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(@Vector(2, f32), undefined)
// @as(f32, undefined)
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), [runtime value])
// @as(@Vector(2, f32), undefined)
// @as(f64, undefined)
// @as(f64, undefined)
// @as(@Vector(2, f64), .{ 6, undefined })
// @as(@Vector(2, f64), .{ undefined, 6 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ 6, undefined })
// @as(@Vector(2, f64), .{ 6, undefined })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ undefined, 6 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ undefined, 6 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(f64, undefined)
// @as(f64, undefined)
// @as(@Vector(2, f64), .{ 0, undefined })
// @as(@Vector(2, f64), .{ undefined, 0 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ 0, undefined })
// @as(@Vector(2, f64), .{ 0, undefined })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ undefined, 0 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ undefined, 0 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(f64, undefined)
// @as(f64, undefined)
// @as(@Vector(2, f64), .{ 9, undefined })
// @as(@Vector(2, f64), .{ undefined, 9 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ 9, undefined })
// @as(@Vector(2, f64), .{ 9, undefined })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ undefined, 9 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), .{ undefined, 9 })
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(f64, undefined)
// @as(@Vector(2, f64), .{ -3, undefined })
// @as(@Vector(2, f64), .{ undefined, -3 })
// @as(@Vector(2, f64), undefined)
// @as(f64, undefined)
// @as(f64, undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(f64, undefined)
// @as(f64, undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(f64, undefined)
// @as(f64, undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(@Vector(2, f64), undefined)
// @as(f64, undefined)
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), [runtime value])
// @as(@Vector(2, f64), undefined)
// @as(f80, undefined)
// @as(f80, undefined)
// @as(@Vector(2, f80), .{ 6, undefined })
// @as(@Vector(2, f80), .{ undefined, 6 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ 6, undefined })
// @as(@Vector(2, f80), .{ 6, undefined })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ undefined, 6 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ undefined, 6 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(f80, undefined)
// @as(f80, undefined)
// @as(@Vector(2, f80), .{ 0, undefined })
// @as(@Vector(2, f80), .{ undefined, 0 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ 0, undefined })
// @as(@Vector(2, f80), .{ 0, undefined })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ undefined, 0 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ undefined, 0 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(f80, undefined)
// @as(f80, undefined)
// @as(@Vector(2, f80), .{ 9, undefined })
// @as(@Vector(2, f80), .{ undefined, 9 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ 9, undefined })
// @as(@Vector(2, f80), .{ 9, undefined })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ undefined, 9 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), .{ undefined, 9 })
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(f80, undefined)
// @as(@Vector(2, f80), .{ -3, undefined })
// @as(@Vector(2, f80), .{ undefined, -3 })
// @as(@Vector(2, f80), undefined)
// @as(f80, undefined)
// @as(f80, undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(f80, undefined)
// @as(f80, undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(f80, undefined)
// @as(f80, undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(@Vector(2, f80), undefined)
// @as(f80, undefined)
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), [runtime value])
// @as(@Vector(2, f80), undefined)
// @as(f128, undefined)
// @as(f128, undefined)
// @as(@Vector(2, f128), .{ 6, undefined })
// @as(@Vector(2, f128), .{ undefined, 6 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ 6, undefined })
// @as(@Vector(2, f128), .{ 6, undefined })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ undefined, 6 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ undefined, 6 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(f128, undefined)
// @as(f128, undefined)
// @as(@Vector(2, f128), .{ 0, undefined })
// @as(@Vector(2, f128), .{ undefined, 0 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ 0, undefined })
// @as(@Vector(2, f128), .{ 0, undefined })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ undefined, 0 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ undefined, 0 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(f128, undefined)
// @as(f128, undefined)
// @as(@Vector(2, f128), .{ 9, undefined })
// @as(@Vector(2, f128), .{ undefined, 9 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ 9, undefined })
// @as(@Vector(2, f128), .{ 9, undefined })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ undefined, 9 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), .{ undefined, 9 })
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(f128, undefined)
// @as(@Vector(2, f128), .{ -3, undefined })
// @as(@Vector(2, f128), .{ undefined, -3 })
// @as(@Vector(2, f128), undefined)
// @as(f128, undefined)
// @as(f128, undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(f128, undefined)
// @as(f128, undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(f128, undefined)
// @as(f128, undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(@Vector(2, f128), undefined)
// @as(f128, undefined)
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), [runtime value])
// @as(@Vector(2, f128), undefined)
