/* -*- Mode: C++; c-default-style: "k&r"; indent-tabs-mode: nil; tab-width: 2; c-basic-offset: 2 -*- */

/* libstaroffice
* Version: MPL 2.0 / LGPLv2+
*
* The contents of this file are subject to the Mozilla Public License Version
* 2.0 (the "License"); you may not use this file except in compliance with
* the License or as specified alternatively below. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
*
* Software distributed under the License is distributed on an "AS IS" basis,
* WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
* for the specific language governing rights and limitations under the
* License.
*
* Major Contributor(s):
* Copyright (C) 2002 William Lachance (wrlach@gmail.com)
* Copyright (C) 2002,2004 Marc Maurer (uwog@uwog.net)
* Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
* Copyright (C) 2006, 2007 Andrew Ziem
* Copyright (C) 2011, 2012 Alonso Laurent (alonso@loria.fr)
*
*
* All Rights Reserved.
*
* For minor contributions see the git repository.
*
* Alternatively, the contents of this file may be used under the terms of
* the GNU Lesser General Public License Version 2 or later (the "LGPLv2+"),
* in which case the provisions of the LGPLv2+ are applicable
* instead of those above.
*/

#include <cstring>
#include <iomanip>
#include <iostream>
#include <limits>
#include <sstream>

#include <librevenge/librevenge.h>

#include "StarEncodingChinese.hxx"
#include "StarEncodingJapanese.hxx"
#include "StarEncodingKorean.hxx"
#include "StarEncodingOtherKorean.hxx"
#include "StarEncodingTradChinese.hxx"

#include "StarEncoding.hxx"

////////////////////////////////////////////////////////////
// constructor/destructor, ...
////////////////////////////////////////////////////////////
StarEncoding::StarEncoding()
{
}

StarEncoding::~StarEncoding()
{
}

bool StarEncoding::convert(std::vector<uint8_t> const &src, StarEncoding::Encoding encoding, std::vector<uint32_t> &dest, std::vector<size_t> &srcPositions)
{
  size_t pos=0;
  while (pos<src.size()) {
    size_t actPos=pos;
    if (!read(src, pos, encoding, dest) && actPos>=pos)
      break;
    srcPositions.resize(dest.size(), actPos);
  }
  return !dest.empty() || src.empty();
}

StarEncoding::Encoding StarEncoding::getEncodingForId(int id)
{
  Encoding res=E_DONTKNOW;
  switch (id) {
  case E_DONTKNOW:
  case E_MS_1252:
  case E_APPLE_ROMAN:
  case E_IBM_437:
  case E_IBM_850:
  case E_IBM_860:
  case E_IBM_861:
  case E_IBM_863:
  case E_IBM_865:
  case E_SYMBOL:
  case E_ASCII_US:
  case E_ISO_8859_1:
  case E_ISO_8859_2:
  case E_ISO_8859_3:
  case E_ISO_8859_4:
  case E_ISO_8859_5:
  case E_ISO_8859_6:
  case E_ISO_8859_7:
  case E_ISO_8859_8:
  case E_ISO_8859_9:
  case E_ISO_8859_14:
  case E_ISO_8859_15:
  case E_IBM_737:
  case E_IBM_775:
  case E_IBM_852:
  case E_IBM_855:
  case E_IBM_857:
  case E_IBM_862:
  case E_IBM_864:
  case E_IBM_866:
  case E_IBM_869:
  case E_MS_874:
  case E_MS_1250:
  case E_MS_1251:
  case E_MS_1253:
  case E_MS_1254:
  case E_MS_1255:
  case E_MS_1256:
  case E_MS_1257:
  case E_MS_1258:
  case E_APPLE_CENTEURO:
  case E_APPLE_CROATIAN:
  case E_APPLE_CYRILLIC:
  case E_APPLE_GREEK:
  case E_APPLE_ICELAND:
  case E_APPLE_ROMANIAN:
  case E_APPLE_TURKISH:
  case E_APPLE_UKRAINIAN:
  case E_APPLE_CHINSIMP:
  case E_APPLE_CHINTRAD:
  case E_APPLE_JAPANESE:
  case E_APPLE_KOREAN:
  case E_MS_932:
  case E_MS_936:
  case E_MS_949:
  case E_MS_950:
  case E_SHIFT_JIS:
  case E_GB_2312:
  case E_GBT_12345:
  case E_GBK:
  case E_BIG5:
  case E_EUC_JP:
  case E_EUC_CN:
  case E_KOI8_R:
  case E_UTF7:
  case E_UTF8:
  case E_ISO_8859_10:
  case E_ISO_8859_13:
  case E_EUC_KR:
  case E_JIS_X_0201:
  case E_JIS_X_0208:
  case E_JIS_X_0212:
  case E_MS_1361:
  case E_BIG5_HKSCS:
  case E_TIS_620:
  case E_KOI8_U:
  case E_ISCII_DEVANAGARI:
  case E_UCS4:
  case E_UCS2:
    res=Encoding(id);
    break;
  default:
    STOFF_DEBUG_MSG(("StarEncoding::getEncodingForId: can not find encoding for id=%d\n", id));
    break;
  }
  if (res==E_DONTKNOW) res=E_MS_1252; // seems to be the default
  // tenccvt.cxx GetExtendedTextEncoding GetExtendedCompatibilityTextEncoding
  if (res==E_ISO_8859_1 || res==E_ISO_8859_15) res=E_MS_1252;
  if (res==E_ISO_8859_2) res=E_MS_1250;
  if (res==E_ISO_8859_5) res=E_MS_1251;
  if (res==E_ISO_8859_7) res=E_MS_1253;
  if (res==E_ISO_8859_9) res=E_MS_1254;
  return res;
}

bool StarEncoding::read
(std::vector<uint8_t> const &src, size_t &pos, StarEncoding::Encoding encoding, std::vector<uint32_t> &dest)
{
  if (encoding==E_SHIFT_JIS || encoding==E_MS_932 || encoding==E_APPLE_JAPANESE)
    return StarEncodingJapanese::readJapanese1(src, pos, encoding, dest);
  if (encoding==E_JIS_X_0208)
    return StarEncodingJapanese::readJapanese208(src, pos, encoding, dest);
  if (encoding==E_JIS_X_0212)
    return StarEncodingJapanese::readJapanese212(src, pos, encoding, dest);
  if (encoding==E_EUC_JP)
    return StarEncodingJapanese::readJapaneseEUC(src, pos, encoding, dest);
  if (encoding==E_BIG5 || encoding==E_MS_950 || encoding==E_APPLE_CHINTRAD)
    return StarEncodingTradChinese::readChinese1(src, pos, encoding, dest);
  if (encoding==E_GBK || encoding==E_GB_2312  || encoding==E_EUC_CN ||
      encoding==E_GBT_12345 || encoding==E_MS_936 || encoding==E_APPLE_CHINSIMP)
    return StarEncodingChinese::readChinese1(src, pos, encoding, dest);
  if (encoding==E_MS_949 || encoding==E_EUC_KR || encoding==E_APPLE_KOREAN)
    return StarEncodingKorean::readKorean1(src, pos, encoding, dest);
  if (encoding==E_BIG5_HKSCS)
    return StarEncodingOtherKorean::readKoreanBig5(src, pos, encoding, dest);
  if (encoding==E_MS_1361)
    return StarEncodingOtherKorean::readKoreanMS1361(src, pos, encoding, dest);
  if (pos>=src.size()) return false;
  auto c=int(src[pos++]);
  auto unicode=uint32_t(c);
  switch (encoding) {
  case E_DONTKNOW: // MS1252 seems to be the default encoding
  case E_ASCII_US: // use the same as MS1252
  case E_MS_1252: {
    if (c<0x80) break;
    static int const val[]= {
      0x20AC,      0, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0x02C6, 0x2030, 0x0160, 0x2039, 0x0152,      0, 0x017D,      0, /* 0x80 */
      0, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0x02DC, 0x2122, 0x0161, 0x203A, 0x0153,      0, 0x017E, 0x0178, /* 0x90 */
      0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF, /* 0xB0 */
      0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7, /* 0xC0 */
      0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7, /* 0xD0 */
      0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF, /* 0xD0 */
      0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7, /* 0xF0 */
      0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_ROMAN: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C4, 0x00C5, 0x00C7, 0x00C9, 0x00D1, 0x00D6, 0x00DC, 0x00E1, /* 0x80 */
      0x00E0, 0x00E2, 0x00E4, 0x00E3, 0x00E5, 0x00E7, 0x00E9, 0x00E8, /* 0x80 */
      0x00EA, 0x00EB, 0x00ED, 0x00EC, 0x00EE, 0x00EF, 0x00F1, 0x00F3, /* 0x90 */
      0x00F2, 0x00F4, 0x00F6, 0x00F5, 0x00FA, 0x00F9, 0x00FB, 0x00FC, /* 0x90 */
      0x2020, 0x00B0, 0x00A2, 0x00A3, 0x00A7, 0x2022, 0x00B6, 0x00DF, /* 0xA0 */
      0x00AE, 0x00A9, 0x2122, 0x00B4, 0x00A8, 0x2260, 0x00C6, 0x00D8, /* 0xA0 */
      0x221E, 0x00B1, 0x2264, 0x2265, 0x00A5, 0x00B5, 0x2202, 0x2211, /* 0xB0 */
      0x220F, 0x03C0, 0x222B, 0x00AA, 0x00BA, 0x03A9, 0x00E6, 0x00F8, /* 0xB0 */
      0x00BF, 0x00A1, 0x00AC, 0x221A, 0x0192, 0x2248, 0x2206, 0x00AB, /* 0xC0 */
      0x00BB, 0x2026, 0x00A0, 0x00C0, 0x00C3, 0x00D5, 0x0152, 0x0153, /* 0xC0 */
      0x2013, 0x2014, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x25CA, /* 0xD0 */
      0x00FF, 0x0178, 0x2044, 0x20AC, 0x2039, 0x203A, 0xFB01, 0xFB02, /* 0xD0 */
      0x2021, 0x00B7, 0x201A, 0x201E, 0x2030, 0x00C2, 0x00CA, 0x00C1, /* 0xE0 */
      0x00CB, 0x00C8, 0x00CD, 0x00CE, 0x00CF, 0x00CC, 0x00D3, 0x00D4, /* 0xE0 */
      0xF8FF, 0x00D2, 0x00DA, 0x00DB, 0x00D9, 0x0131, 0x02C6, 0x02DC, /* 0xF0 */
      0x00AF, 0x02D8, 0x02D9, 0x02DA, 0x00B8, 0x02DD, 0x02DB, 0x02C7  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_437: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7, /* 0x80 */
      0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x00EC, 0x00C4, 0x00C5, /* 0x80 */
      0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9, /* 0x90 */
      0x00FF, 0x00D6, 0x00DC, 0x00A2, 0x00A3, 0x00A5, 0x20A7, 0x0192, /* 0x90 */
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA, /* 0xA0 */
      0x00BF, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xB0 */
      0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567, /* 0xC0 */
      0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B, /* 0xD0 */
      0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x03B1, 0x00DF, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4, /* 0xE0 */
      0x03A6, 0x0398, 0x03A9, 0x03B4, 0x221E, 0x03C6, 0x03B5, 0x2229, /* 0xE0 */
      0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_850: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7, /* 0x80 */
      0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x00EC, 0x00C4, 0x00C5, /* 0x80 */
      0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9, /* 0x90 */
      0x00FF, 0x00D6, 0x00DC, 0x00F8, 0x00A3, 0x00D8, 0x00D7, 0x0192, /* 0x90 */
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA, /* 0xA0 */
      0x00BF, 0x00AE, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00C1, 0x00C2, 0x00C0, /* 0xB0 */
      0x00A9, 0x2563, 0x2551, 0x2557, 0x255D, 0x00A2, 0x00A5, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x00E3, 0x00C3, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x00A4, /* 0xC0 */
      0x00F0, 0x00D0, 0x00CA, 0x00CB, 0x00C8, 0x20AC, 0x00CD, 0x00CE, /* 0xD0 */
      0x00CF, 0x2518, 0x250C, 0x2588, 0x2584, 0x00A6, 0x00CC, 0x2580, /* 0xD0 */
      0x00D3, 0x00DF, 0x00D4, 0x00D2, 0x00F5, 0x00D5, 0x00B5, 0x00FE, /* 0xE0 */
      0x00DE, 0x00DA, 0x00DB, 0x00D9, 0x00FD, 0x00DD, 0x00AF, 0x00B4, /* 0xE0 */
      0x00AD, 0x00B1, 0x2017, 0x00BE, 0x00B6, 0x00A7, 0x00F7, 0x00B8, /* 0xF0 */
      0x00B0, 0x00A8, 0x00B7, 0x00B9, 0x00B3, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_860: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E3, 0x00E0, 0x00C1, 0x00E7, /* 0x80 */
      0x00EA, 0x00CA, 0x00E8, 0x00CD, 0x00D4, 0x00EC, 0x00C3, 0x00C2, /* 0x80 */
      0x00C9, 0x00C0, 0x00C8, 0x00F4, 0x00F5, 0x00F2, 0x00DA, 0x00F9, /* 0x90 */
      0x00CC, 0x00D5, 0x00DC, 0x00A2, 0x00A3, 0x00D9, 0x20A7, 0x00D3, /* 0x90 */
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA, /* 0xA0 */
      0x00BF, 0x00D2, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xB0 */
      0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567, /* 0xC0 */
      0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B, /* 0xD0 */
      0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x03B1, 0x00DF, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4, /* 0xE0 */
      0x03A6, 0x0398, 0x03A9, 0x03B4, 0x221E, 0x03C6, 0x03B5, 0x2229, /* 0xE0 */
      0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_861: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7, /* 0x80 */
      0x00EA, 0x00EB, 0x00E8, 0x00D0, 0x00F0, 0x00DE, 0x00C4, 0x00C5, /* 0x80 */
      0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00FE, 0x00FB, 0x00DD, /* 0x90 */
      0x00FD, 0x00D6, 0x00DC, 0x00F8, 0x00A3, 0x00D8, 0x20A7, 0x0192, /* 0x90 */
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00C1, 0x00CD, 0x00D3, 0x00DA, /* 0xA0 */
      0x00BF, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xB0 */
      0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567, /* 0xC0 */
      0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B, /* 0xD0 */
      0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x03B1, 0x00DF, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4, /* 0xE0 */
      0x03A6, 0x0398, 0x03A9, 0x03B4, 0x221E, 0x03C6, 0x03B5, 0x2229, /* 0xE0 */
      0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_863: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00C2, 0x00E0, 0x00B6, 0x00E7, /* 0x80 */
      0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x2017, 0x00C0, 0x00A7, /* 0x80 */
      0x00C9, 0x00C8, 0x00CA, 0x00F4, 0x00CB, 0x00CF, 0x00FB, 0x00F9, /* 0x90 */
      0x00A4, 0x00D4, 0x00DC, 0x00A2, 0x00A3, 0x00D9, 0x00DB, 0x0192, /* 0x90 */
      0x00A6, 0x00B4, 0x00F3, 0x00FA, 0x00A8, 0x00B8, 0x00B3, 0x00AF, /* 0xA0 */
      0x00CE, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00BE, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xB0 */
      0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567, /* 0xC0 */
      0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B, /* 0xD0 */
      0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x03B1, 0x00DF, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4, /* 0xE0 */
      0x03A6, 0x0398, 0x03A9, 0x03B4, 0x221E, 0x03C6, 0x03B5, 0x2229, /* 0xE0 */
      0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_865: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7, /* 0x80 */
      0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x00EC, 0x00C4, 0x00C5, /* 0x80 */
      0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9, /* 0x90 */
      0x00FF, 0x00D6, 0x00DC, 0x00F8, 0x00A3, 0x00D8, 0x20A7, 0x0192, /* 0x90 */
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA, /* 0xA0 */
      0x00BF, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00A4, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xB0 */
      0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567, /* 0xC0 */
      0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B, /* 0xD0 */
      0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x03B1, 0x00DF, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4, /* 0xE0 */
      0x03A6, 0x0398, 0x03A9, 0x03B4, 0x221E, 0x03C6, 0x03B5, 0x2229, /* 0xE0 */
      0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_SYMBOL: {
    if (c<0x1f) break;
    unicode=static_cast<uint32_t>(0xF000+c);
    break;
  }
  case E_ISO_8859_1: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF, /* 0xB0 */
      0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7, /* 0xC0 */
      0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7, /* 0xD0 */
      0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF, /* 0xD0 */
      0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7, /* 0xF0 */
      0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_2: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x0104, 0x02D8, 0x0141, 0x00A4, 0x013D, 0x015A, 0x00A7, /* 0xA0 */
      0x00A8, 0x0160, 0x015E, 0x0164, 0x0179, 0x00AD, 0x017D, 0x017B, /* 0xA0 */
      0x00B0, 0x0105, 0x02DB, 0x0142, 0x00B4, 0x013E, 0x015B, 0x02C7, /* 0xB0 */
      0x00B8, 0x0161, 0x015F, 0x0165, 0x017A, 0x02DD, 0x017E, 0x017C, /* 0xB0 */
      0x0154, 0x00C1, 0x00C2, 0x0102, 0x00C4, 0x0139, 0x0106, 0x00C7, /* 0xC0 */
      0x010C, 0x00C9, 0x0118, 0x00CB, 0x011A, 0x00CD, 0x00CE, 0x010E, /* 0xC0 */
      0x0110, 0x0143, 0x0147, 0x00D3, 0x00D4, 0x0150, 0x00D6, 0x00D7, /* 0xD0 */
      0x0158, 0x016E, 0x00DA, 0x0170, 0x00DC, 0x00DD, 0x0162, 0x00DF, /* 0xD0 */
      0x0155, 0x00E1, 0x00E2, 0x0103, 0x00E4, 0x013A, 0x0107, 0x00E7, /* 0xE0 */
      0x010D, 0x00E9, 0x0119, 0x00EB, 0x011B, 0x00ED, 0x00EE, 0x010F, /* 0xE0 */
      0x0111, 0x0144, 0x0148, 0x00F3, 0x00F4, 0x0151, 0x00F6, 0x00F7, /* 0xF0 */
      0x0159, 0x016F, 0x00FA, 0x0171, 0x00FC, 0x00FD, 0x0163, 0x02D9  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_3: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x0126, 0x02D8, 0x00A3, 0x00A4,      0, 0x0124, 0x00A7, /* 0xA0 */
      0x00A8, 0x0130, 0x015E, 0x011E, 0x0134, 0x00AD,      0, 0x017B, /* 0xA0 */
      0x00B0, 0x0127, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x0125, 0x00B7, /* 0xB0 */
      0x00B8, 0x0131, 0x015F, 0x011F, 0x0135, 0x00BD,      0, 0x017C, /* 0xB0 */
      0x00C0, 0x00C1, 0x00C2,      0, 0x00C4, 0x010A, 0x0108, 0x00C7, /* 0xC0 */
      0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x0120, 0x00D6, 0x00D7, /* 0xD0 */
      0x011C, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x016C, 0x015C, 0x00DF, /* 0xD0 */
      0x00E0, 0x00E1, 0x00E2,      0, 0x00E4, 0x010B, 0x0109, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x0121, 0x00F6, 0x00F7, /* 0xF0 */
      0x011D, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x016D, 0x015D, 0x02D9  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_4: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x0104, 0x0138, 0x0156, 0x00A4, 0x0128, 0x013B, 0x00A7, /* 0xA0 */
      0x00A8, 0x0160, 0x0112, 0x0122, 0x0166, 0x00AD, 0x017D, 0x00AF, /* 0xA0 */
      0x00B0, 0x0105, 0x02DB, 0x0157, 0x00B4, 0x0129, 0x013C, 0x02C7, /* 0xB0 */
      0x00B8, 0x0161, 0x0113, 0x0123, 0x0167, 0x014A, 0x017E, 0x014B, /* 0xB0 */
      0x0100, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x012E, /* 0xC0 */
      0x010C, 0x00C9, 0x0118, 0x00CB, 0x0116, 0x00CD, 0x00CE, 0x012A, /* 0xC0 */
      0x0110, 0x0145, 0x014C, 0x0136, 0x00D4, 0x00D5, 0x00D6, 0x00D7, /* 0xD0 */
      0x00D8, 0x0172, 0x00DA, 0x00DB, 0x00DC, 0x0168, 0x016A, 0x00DF, /* 0xD0 */
      0x0101, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x012F, /* 0xE0 */
      0x010D, 0x00E9, 0x0119, 0x00EB, 0x0117, 0x00ED, 0x00EE, 0x012B, /* 0xE0 */
      0x0111, 0x0146, 0x014D, 0x0137, 0x00F4, 0x00F5, 0x00F6, 0x00F7, /* 0xF0 */
      0x00F8, 0x0173, 0x00FA, 0x00FB, 0x00FC, 0x0169, 0x016B, 0x02D9  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_5: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x0401, 0x0402, 0x0403, 0x0404, 0x0405, 0x0406, 0x0407, /* 0xA0 */
      0x0408, 0x0409, 0x040A, 0x040B, 0x040C, 0x00AD, 0x040E, 0x040F, /* 0xA0 */
      0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0xB0 */
      0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F, /* 0xB0 */
      0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0xC0 */
      0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 0x042F, /* 0xC0 */
      0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xD0 */
      0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F, /* 0xD0 */
      0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xE0 */
      0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F, /* 0xE0 */
      0x2116, 0x0451, 0x0452, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457, /* 0xF0 */
      0x0458, 0x0459, 0x045A, 0x045B, 0x045C, 0x00A7, 0x045E, 0x045F  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_6: {
    if (c<0xa0 || c>0xf2) break;
    static int const val[]= {
      0x00A0,      0,      0,      0, 0x00A4,      0,      0,      0, /* 0xA0 */
      0,      0,      0,      0, 0x060C, 0x00AD,      0,      0, /* 0xA0 */
      0,      0,      0,      0,      0,      0,      0,      0, /* 0xB0 */
      0,      0,      0, 0x061B,      0,      0,      0, 0x061F, /* 0xB0 */
      0, 0x0621, 0x0622, 0x0623, 0x0624, 0x0625, 0x0626, 0x0627, /* 0xC0 */
      0x0628, 0x0629, 0x062A, 0x062B, 0x062C, 0x062D, 0x062E, 0x062F, /* 0xC0 */
      0x0630, 0x0631, 0x0632, 0x0633, 0x0634, 0x0635, 0x0636, 0x0637, /* 0xD0 */
      0x0638, 0x0639, 0x063A,      0,      0,      0,      0,      0, /* 0xD0 */
      0x0640, 0x0641, 0x0642, 0x0643, 0x0644, 0x0645, 0x0646, 0x0647, /* 0xE0 */
      0x0648, 0x0649, 0x064A, 0x064B, 0x064C, 0x064D, 0x064E, 0x064F, /* 0xE0 */
      0x0650, 0x0651, 0x0652                                          /* 0xF0 */
      /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_7: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x2018, 0x2019, 0x00A3,      0,      0, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9,      0, 0x00AB, 0x00AC, 0x00AD,      0, 0x2015, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x0384, 0x0385, 0x0386, 0x00B7, /* 0xB0 */
      0x0388, 0x0389, 0x038A, 0x00BB, 0x038C, 0x00BD, 0x038E, 0x038F, /* 0xB0 */
      0x0390, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, /* 0xC0 */
      0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F, /* 0xC0 */
      0x03A0, 0x03A1,      0, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7, /* 0xD0 */
      0x03A8, 0x03A9, 0x03AA, 0x03AB, 0x03AC, 0x03AD, 0x03AE, 0x03AF, /* 0xD0 */
      0x03B0, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, /* 0xE0 */
      0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF, /* 0xE0 */
      0x03C0, 0x03C1, 0x03C2, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7, /* 0xF0 */
      0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03CC, 0x03CD, 0x03CE,      0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_8: {
    if (c<0xa0 || c>0xfe) break;
    static int const val[]= {
      0x00A0,      0, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x00D7, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x00B9, 0x00F7, 0x00BB, 0x00BC, 0x00BD, 0x00BE,      0, /* 0xB0 */
      0,      0,      0,      0,      0,      0,      0,      0, /* 0xC0 */
      0,      0,      0,      0,      0,      0,      0,      0, /* 0xC0 */
      0,      0,      0,      0,      0,      0,      0,      0, /* 0xD0 */
      0,      0,      0,      0,      0,      0,      0, 0x2017, /* 0xD0 */
      0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5, 0x05D6, 0x05D7, /* 0xE0 */
      0x05D8, 0x05D9, 0x05DA, 0x05DB, 0x05DC, 0x05DD, 0x05DE, 0x05DF, /* 0xE0 */
      0x05E0, 0x05E1, 0x05E2, 0x05E3, 0x05E4, 0x05E5, 0x05E6, 0x05E7, /* 0xF0 */
      0x05E8, 0x05E9, 0x05EA,      0,      0, 0x200E, 0x200F          /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_9: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF, /* 0xB0 */
      0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7, /* 0xC0 */
      0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0x011E, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7, /* 0xD0 */
      0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x0130, 0x015E, 0x00DF, /* 0xD0 */
      0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0x011F, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7, /* 0xF0 */
      0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x0131, 0x015F, 0x00FF  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_10: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x0104, 0x0112, 0x0122, 0x012A, 0x0128, 0x0136, 0x00A7, /* 0xA0 */
      0x013B, 0x0110, 0x0160, 0x0166, 0x017D, 0x00AD, 0x016A, 0x014A, /* 0xA0 */
      0x00B0, 0x0105, 0x0113, 0x0123, 0x012B, 0x0129, 0x0137, 0x00B7, /* 0xB0 */
      0x013C, 0x0111, 0x0161, 0x0167, 0x017E, 0x2015, 0x016B, 0x014B, /* 0xB0 */
      0x0100, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x012E, /* 0xC0 */
      0x010C, 0x00C9, 0x0118, 0x00CB, 0x0116, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0x00D0, 0x0145, 0x014C, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x0168, /* 0xD0 */
      0x00D8, 0x0172, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF, /* 0xD0 */
      0x0101, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x012F, /* 0xE0 */
      0x010D, 0x00E9, 0x0119, 0x00EB, 0x0117, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0x00F0, 0x0146, 0x014D, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x0169, /* 0xF0 */
      0x00F8, 0x0173, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x0138  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_13: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x201D, 0x00A2, 0x00A3, 0x00A4, 0x201E, 0x00A6, 0x00A7, /* 0xA0 */
      0x00D8, 0x00A9, 0x0156, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00C6, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x201C, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00F8, 0x00B9, 0x0157, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00E6, /* 0xB0 */
      0x0104, 0x012E, 0x0100, 0x0106, 0x00C4, 0x00C5, 0x0118, 0x0112, /* 0xC0 */
      0x010C, 0x00C9, 0x0179, 0x0116, 0x0122, 0x0136, 0x012A, 0x013B, /* 0xC0 */
      0x0160, 0x0143, 0x0145, 0x00D3, 0x014C, 0x00D5, 0x00D6, 0x00D7, /* 0xD0 */
      0x0172, 0x0141, 0x015A, 0x016A, 0x00DC, 0x017B, 0x017D, 0x00DF, /* 0xD0 */
      0x0105, 0x012F, 0x0101, 0x0107, 0x00E4, 0x00E5, 0x0119, 0x0113, /* 0xE0 */
      0x010D, 0x00E9, 0x017A, 0x0117, 0x0123, 0x0137, 0x012B, 0x013C, /* 0xE0 */
      0x0161, 0x0144, 0x0146, 0x00F3, 0x014D, 0x00F5, 0x00F6, 0x00F7, /* 0xF0 */
      0x0173, 0x0142, 0x015B, 0x016B, 0x00FC, 0x017C, 0x017E, 0x2019  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_14: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x1E02, 0x1E03, 0x00A3, 0x010A, 0x010B, 0x1E0A, 0x00A7, /* 0xA0 */
      0x1E80, 0x00A9, 0x1E82, 0x1E0B, 0x1EF2, 0x00AD, 0x00AE, 0x0178, /* 0xA0 */
      0x1E1E, 0x1E1F, 0x0120, 0x0121, 0x1E40, 0x1E41, 0x00B6, 0x1E56, /* 0xB0 */
      0x1E81, 0x1E57, 0x1E83, 0x1E60, 0x1EF3, 0x1E84, 0x1E85, 0x1E61, /* 0xB0 */
      0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7, /* 0xC0 */
      0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0x0174, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x1E6A, /* 0xD0 */
      0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x0176, 0x00DF, /* 0xD0 */
      0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0x0175, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x1E6B, /* 0xF0 */
      0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x0177, 0x00FF  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_ISO_8859_15: {
    if (c<0xa0) break;
    static int const val[]= {
      0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x20AC, 0x00A5, 0x0160, 0x00A7, /* 0xA0 */
      0x0161, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x017D, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x017E, 0x00B9, 0x00BA, 0x00BB, 0x0152, 0x0153, 0x0178, 0x00BF, /* 0xB0 */
      0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7, /* 0xC0 */
      0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7, /* 0xD0 */
      0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF, /* 0xD0 */
      0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7, /* 0xF0 */
      0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_IBM_737: {
    if (c<0x80) break;
    static int const val[]= {
      0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, 0x0398, /* 0x80 */
      0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F, 0x03A0, /* 0x80 */
      0x03A1, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7, 0x03A8, 0x03A9, /* 0x90 */
      0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, 0x03B8, /* 0x90 */
      0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF, 0x03C0, /* 0xA0 */
      0x03C1, 0x03C3, 0x03C2, 0x03C4, 0x03C5, 0x03C6, 0x03C7, 0x03C8, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xB0 */
      0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567, /* 0xC0 */
      0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B, /* 0xD0 */
      0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x03C9, 0x03AC, 0x03AD, 0x03AE, 0x03CA, 0x03AF, 0x03CC, 0x03CD, /* 0xE0 */
      0x03CB, 0x03CE, 0x0386, 0x0388, 0x0389, 0x038A, 0x038C, 0x038E, /* 0xE0 */
      0x038F, 0x00B1, 0x2265, 0x2264, 0x03AA, 0x03AB, 0x00F7, 0x2248, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_775: {
    if (c<0x80) break;
    static int const val[]= {
      0x0106, 0x00FC, 0x00E9, 0x0101, 0x00E4, 0x0123, 0x00E5, 0x0107, /* 0x80 */
      0x0142, 0x0113, 0x0156, 0x0157, 0x012B, 0x0179, 0x00C4, 0x00C5, /* 0x80 */
      0x00C9, 0x00E6, 0x00C6, 0x014D, 0x00F6, 0x0122, 0x00A2, 0x015A, /* 0x90 */
      0x015B, 0x00D6, 0x00DC, 0x00F8, 0x00A3, 0x00D8, 0x00D7, 0x00A4, /* 0x90 */
      0x0100, 0x012A, 0x00F3, 0x017B, 0x017C, 0x017A, 0x201D, 0x00A6, /* 0xA0 */
      0x00A9, 0x00AE, 0x00AC, 0x00BD, 0x00BC, 0x0141, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x0104, 0x010C, 0x0118, /* 0xB0 */
      0x0116, 0x2563, 0x2551, 0x2557, 0x255D, 0x012E, 0x0160, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x0172, 0x016A, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x017D, /* 0xC0 */
      0x0105, 0x010D, 0x0119, 0x0117, 0x012F, 0x0161, 0x0173, 0x016B, /* 0xD0 */
      0x017E, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x00D3, 0x00DF, 0x014C, 0x0143, 0x00F5, 0x00D5, 0x00B5, 0x0144, /* 0xE0 */
      0x0136, 0x0137, 0x013B, 0x013C, 0x0146, 0x0112, 0x0145, 0x2019, /* 0xE0 */
      0x00AD, 0x00B1, 0x201C, 0x00BE, 0x00B6, 0x00A7, 0x00F7, 0x201E, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x00B9, 0x00B3, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_852: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x016F, 0x0107, 0x00E7, /* 0x80 */
      0x0142, 0x00EB, 0x0150, 0x0151, 0x00EE, 0x0179, 0x00C4, 0x0106, /* 0x80 */
      0x00C9, 0x0139, 0x013A, 0x00F4, 0x00F6, 0x013D, 0x013E, 0x015A, /* 0x90 */
      0x015B, 0x00D6, 0x00DC, 0x0164, 0x0165, 0x0141, 0x00D7, 0x010D, /* 0x90 */
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x0104, 0x0105, 0x017D, 0x017E, /* 0xA0 */
      0x0118, 0x0119, 0x00AC, 0x017A, 0x010C, 0x015F, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00C1, 0x00C2, 0x011A, /* 0xB0 */
      0x015E, 0x2563, 0x2551, 0x2557, 0x255D, 0x017B, 0x017C, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x0102, 0x0103, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x00A4, /* 0xC0 */
      0x0111, 0x0110, 0x010E, 0x00CB, 0x010F, 0x0147, 0x00CD, 0x00CE, /* 0xD0 */
      0x011B, 0x2518, 0x250C, 0x2588, 0x2584, 0x0162, 0x016E, 0x2580, /* 0xD0 */
      0x00D3, 0x00DF, 0x00D4, 0x0143, 0x0144, 0x0148, 0x0160, 0x0161, /* 0xE0 */
      0x0154, 0x00DA, 0x0155, 0x0170, 0x00FD, 0x00DD, 0x0163, 0x00B4, /* 0xE0 */
      0x00AD, 0x02DD, 0x02DB, 0x02C7, 0x02D8, 0x00A7, 0x00F7, 0x00B8, /* 0xF0 */
      0x00B0, 0x00A8, 0x02D9, 0x0171, 0x0158, 0x0159, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_855: {
    if (c<0x80) break;
    static int const val[]= {
      0x0452, 0x0402, 0x0453, 0x0403, 0x0451, 0x0401, 0x0454, 0x0404, /* 0x80 */
      0x0455, 0x0405, 0x0456, 0x0406, 0x0457, 0x0407, 0x0458, 0x0408, /* 0x80 */
      0x0459, 0x0409, 0x045A, 0x040A, 0x045B, 0x040B, 0x045C, 0x040C, /* 0x90 */
      0x045E, 0x040E, 0x045F, 0x040F, 0x044E, 0x042E, 0x044A, 0x042A, /* 0x90 */
      0x0430, 0x0410, 0x0431, 0x0411, 0x0446, 0x0426, 0x0434, 0x0414, /* 0xA0 */
      0x0435, 0x0415, 0x0444, 0x0424, 0x0433, 0x0413, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x0445, 0x0425, 0x0438, /* 0xB0 */
      0x0418, 0x2563, 0x2551, 0x2557, 0x255D, 0x0439, 0x0419, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x043A, 0x041A, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x00A4, /* 0xC0 */
      0x043B, 0x041B, 0x043C, 0x041C, 0x043D, 0x041D, 0x043E, 0x041E, /* 0xD0 */
      0x043F, 0x2518, 0x250C, 0x2588, 0x2584, 0x041F, 0x044F, 0x2580, /* 0xD0 */
      0x042F, 0x0440, 0x0420, 0x0441, 0x0421, 0x0442, 0x0422, 0x0443, /* 0xE0 */
      0x0423, 0x0436, 0x0416, 0x0432, 0x0412, 0x044C, 0x042C, 0x2116, /* 0xE0 */
      0x00AD, 0x044B, 0x042B, 0x0437, 0x0417, 0x0448, 0x0428, 0x044D, /* 0xF0 */
      0x042D, 0x0449, 0x0429, 0x0447, 0x0427, 0x00A7, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_857: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7, /* 0x80 */
      0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x0131, 0x00C4, 0x00C5, /* 0x80 */
      0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9, /* 0x90 */
      0x0130, 0x00D6, 0x00DC, 0x00F8, 0x00A3, 0x00D8, 0x015E, 0x015F, /* 0x90 */
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x011E, 0x011F, /* 0xA0 */
      0x00BF, 0x00AE, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00C1, 0x00C2, 0x00C0, /* 0xB0 */
      0x00A9, 0x2563, 0x2551, 0x2557, 0x255D, 0x00A2, 0x00A5, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x00E3, 0x00C3, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x00A4, /* 0xC0 */
      0x00BA, 0x00AA, 0x00CA, 0x00CB, 0x00C8,      0, 0x00CD, 0x00CE, /* 0xD0 */
      0x00CF, 0x2518, 0x250C, 0x2588, 0x2584, 0x00A6, 0x00CC, 0x2580, /* 0xD0 */
      0x00D3, 0x00DF, 0x00D4, 0x00D2, 0x00F5, 0x00D5, 0x00B5,      0, /* 0xE0 */
      0x00D7, 0x00DA, 0x00DB, 0x00D9, 0x00EC, 0x00FF, 0x00AF, 0x00B4, /* 0xE0 */
      0x00AD, 0x00B1,      0, 0x00BE, 0x00B6, 0x00A7, 0x00F7, 0x00B8, /* 0xF0 */
      0x00B0, 0x00A8, 0x00B7, 0x00B9, 0x00B3, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_862: {
    if (c<0x80) break;
    static int const val[]= {
      0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5, 0x05D6, 0x05D7, /* 0x80 */
      0x05D8, 0x05D9, 0x05DA, 0x05DB, 0x05DC, 0x05DD, 0x05DE, 0x05DF, /* 0x80 */
      0x05E0, 0x05E1, 0x05E2, 0x05E3, 0x05E4, 0x05E5, 0x05E6, 0x05E7, /* 0x90 */
      0x05E8, 0x05E9, 0x05EA, 0x00A2, 0x00A3, 0x00A5, 0x20A7, 0x0192, /* 0x90 */
      0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA, /* 0xA0 */
      0x00BF, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xB0 */
      0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567, /* 0xC0 */
      0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B, /* 0xD0 */
      0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x03B1, 0x00DF, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4, /* 0xE0 */
      0x03A6, 0x0398, 0x03A9, 0x03B4, 0x221E, 0x03C6, 0x03B5, 0x2229, /* 0xE0 */
      0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_864: {
    if (c<0x80) break;
    static int const val[]= {
      0x00B0, 0x00B7, 0x2219, 0x221A, 0x2592, 0x2500, 0x2502, 0x253C, /* 0x80 */
      0x2524, 0x252C, 0x251C, 0x2534, 0x2510, 0x250C, 0x2514, 0x2518, /* 0x80 */
      0x03B2, 0x221E, 0x03C6, 0x00B1, 0x00BD, 0x00BC, 0x2248, 0x00AB, /* 0x90 */
      0x00BB, 0xFEF7, 0xFEF8,      0,      0, 0xFEFB, 0xFEFC,      0, /* 0x90 */
      0x00A0, 0x00AD, 0xFE82, 0x00A3, 0x00A4, 0xFE84,      0,      0, /* 0xA0 */
      0xFE8E, 0xFE8F, 0xFE95, 0xFE99, 0x060C, 0xFE9D, 0xFEA1, 0xFEA5, /* 0xA0 */
      0x0660, 0x0661, 0x0662, 0x0663, 0x0664, 0x0665, 0x0666, 0x0667, /* 0xB0 */
      0x0668, 0x0669, 0xFED1, 0x061B, 0xFEB1, 0xFEB5, 0xFEB9, 0x061F, /* 0xB0 */
      0x00A2, 0xFE80, 0xFE81, 0xFE83, 0xFE85, 0xFECA, 0xFE8B, 0xFE8D, /* 0xC0 */
      0xFE91, 0xFE93, 0xFE97, 0xFE9B, 0xFE9F, 0xFEA3, 0xFEA7, 0xFEA9, /* 0xC0 */
      0xFEAB, 0xFEAD, 0xFEAF, 0xFEB3, 0xFEB7, 0xFEBB, 0xFEBF, 0xFEC1, /* 0xD0 */
      0xFEC5, 0xFECB, 0xFECF, 0x00A6, 0x00AC, 0x00F7, 0x00D7, 0xFEC9, /* 0xD0 */
      0x0640, 0xFED3, 0xFED7, 0xFEDB, 0xFEDF, 0xFEE3, 0xFEE7, 0xFEEB, /* 0xE0 */
      0xFEED, 0xFEEF, 0xFEF3, 0xFEBD, 0xFECC, 0xFECE, 0xFECD, 0xFEE1, /* 0xE0 */
      0xFE7D, 0x0651, 0xFEE5, 0xFEE9, 0xFEEC, 0xFEF0, 0xFEF2, 0xFED0, /* 0xF0 */
      0xFED5, 0xFEF5, 0xFEF6, 0xFEDD, 0xFED9, 0xFEF1, 0x25A0,      0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_866: {
    if (c<0x80) break;
    static int const val[]= {
      0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0x80 */
      0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F, /* 0x80 */
      0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0x90 */
      0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 0x042F, /* 0x90 */
      0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xA0 */
      0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xB0 */
      0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567, /* 0xC0 */
      0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B, /* 0xD0 */
      0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580, /* 0xD0 */
      0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xE0 */
      0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F, /* 0xE0 */
      0x0401, 0x0451, 0x0404, 0x0454, 0x0407, 0x0457, 0x040E, 0x045E, /* 0xF0 */
      0x00B0, 0x2219, 0x00B7, 0x221A, 0x2116, 0x00A4, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_IBM_869: {
    if (c<0x86) break;
    static int const val[]= {
      0x0386,      0, /* 0x80 */
      0x00B7, 0x00AC, 0x00A6, 0x2018, 0x2019, 0x0388, 0x2015, 0x0389, /* 0x80 */
      0x038A, 0x03AA, 0x038C,      0,      0, 0x038E, 0x03AB, 0x00A9, /* 0x90 */
      0x038F, 0x00B2, 0x00B3, 0x03AC, 0x00A3, 0x03AD, 0x03AE, 0x03AF, /* 0x90 */
      0x03CA, 0x0390, 0x03CC, 0x03CD, 0x0391, 0x0392, 0x0393, 0x0394, /* 0xA0 */
      0x0395, 0x0396, 0x0397, 0x00BD, 0x0398, 0x0399, 0x00AB, 0x00BB, /* 0xA0 */
      0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x039A, 0x039B, 0x039C, /* 0xB0 */
      0x039D, 0x2563, 0x2551, 0x2557, 0x255D, 0x039E, 0x039F, 0x2510, /* 0xB0 */
      0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x03A0, 0x03A1, /* 0xC0 */
      0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x03A3, /* 0xC0 */
      0x03A4, 0x03A5, 0x03A6, 0x03A7, 0x03A8, 0x03A9, 0x03B1, 0x03B2, /* 0xD0 */
      0x03B3, 0x2518, 0x250C, 0x2588, 0x2584, 0x03B4, 0x03B5, 0x2580, /* 0xD0 */
      0x03B6, 0x03B7, 0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, /* 0xE0 */
      0x03BE, 0x03BF, 0x03C0, 0x03C1, 0x03C3, 0x03C2, 0x03C4, 0x0384, /* 0xE0 */
      0x00AD, 0x00B1, 0x03C5, 0x03C6, 0x03C7, 0x00A7, 0x03C8, 0x0385, /* 0xF0 */
      0x00B0, 0x00A8, 0x03C9, 0x03CB, 0x03B0, 0x03CE, 0x25A0, 0x00A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x86]);
    break;
  }
  case E_TIS_620: // checkme
  case E_MS_874: {
    if (c<0xa0 || c>0xfb) break;
    static int const val[]= {
      0x00A0, 0x0E01, 0x0E02, 0x0E03, 0x0E04, 0x0E05, 0x0E06, 0x0E07, /* 0xA0 */
      0x0E08, 0x0E09, 0x0E0A, 0x0E0B, 0x0E0C, 0x0E0D, 0x0E0E, 0x0E0F, /* 0xA0 */
      0x0E10, 0x0E11, 0x0E12, 0x0E13, 0x0E14, 0x0E15, 0x0E16, 0x0E17, /* 0xB0 */
      0x0E18, 0x0E19, 0x0E1A, 0x0E1B, 0x0E1C, 0x0E1D, 0x0E1E, 0x0E1F, /* 0xB0 */
      0x0E20, 0x0E21, 0x0E22, 0x0E23, 0x0E24, 0x0E25, 0x0E26, 0x0E27, /* 0xC0 */
      0x0E28, 0x0E29, 0x0E2A, 0x0E2B, 0x0E2C, 0x0E2D, 0x0E2E, 0x0E2F, /* 0xC0 */
      0x0E30, 0x0E31, 0x0E32, 0x0E33, 0x0E34, 0x0E35, 0x0E36, 0x0E37, /* 0xD0 */
      0x0E38, 0x0E39, 0x0E3A,      0,      0,      0,      0, 0x0E3F, /* 0xD0 */
      0x0E40, 0x0E41, 0x0E42, 0x0E43, 0x0E44, 0x0E45, 0x0E46, 0x0E47, /* 0xE0 */
      0x0E48, 0x0E49, 0x0E4A, 0x0E4B, 0x0E4C, 0x0E4D, 0x0E4E, 0x0E4F, /* 0xE0 */
      0x0E50, 0x0E51, 0x0E52, 0x0E53, 0x0E54, 0x0E55, 0x0E56, 0x0E57, /* 0xF0 */
      0x0E58, 0x0E59, 0x0E5A, 0x0E5B                                  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0xa0]);
    break;
  }
  case E_MS_1250: {
    if (c<0x80) break;
    static int const val[]= {
      0x20AC,      0, 0x201A,      0, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0, 0x2030, 0x0160, 0x2039, 0x015A, 0x0164, 0x017D, 0x0179, /* 0x80 */
      0, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0, 0x2122, 0x0161, 0x203A, 0x015B, 0x0165, 0x017E, 0x017A, /* 0x90 */
      0x00A0, 0x02C7, 0x02D8, 0x0141, 0x00A4, 0x0104, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x015E, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x017B, /* 0xA0 */
      0x00B0, 0x00B1, 0x02DB, 0x0142, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x0105, 0x015F, 0x00BB, 0x013D, 0x02DD, 0x013E, 0x017C, /* 0xB0 */
      0x0154, 0x00C1, 0x00C2, 0x0102, 0x00C4, 0x0139, 0x0106, 0x00C7, /* 0xC0 */
      0x010C, 0x00C9, 0x0118, 0x00CB, 0x011A, 0x00CD, 0x00CE, 0x010E, /* 0xC0 */
      0x0110, 0x0143, 0x0147, 0x00D3, 0x00D4, 0x0150, 0x00D6, 0x00D7, /* 0xD0 */
      0x0158, 0x016E, 0x00DA, 0x0170, 0x00DC, 0x00DD, 0x0162, 0x00DF, /* 0xD0 */
      0x0155, 0x00E1, 0x00E2, 0x0103, 0x00E4, 0x013A, 0x0107, 0x00E7, /* 0xE0 */
      0x010D, 0x00E9, 0x0119, 0x00EB, 0x011B, 0x00ED, 0x00EE, 0x010F, /* 0xE0 */
      0x0111, 0x0144, 0x0148, 0x00F3, 0x00F4, 0x0151, 0x00F6, 0x00F7, /* 0xF0 */
      0x0159, 0x016F, 0x00FA, 0x0171, 0x00FC, 0x00FD, 0x0163, 0x02D9  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_MS_1251: {
    if (c<0x80) break;
    static int const val[]= {
      0x0402, 0x0403, 0x201A, 0x0453, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0x20AC, 0x2030, 0x0409, 0x2039, 0x040A, 0x040C, 0x040B, 0x040F, /* 0x80 */
      0x0452, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0, 0x2122, 0x0459, 0x203A, 0x045A, 0x045C, 0x045B, 0x045F, /* 0x90 */
      0x00A0, 0x040E, 0x045E, 0x0408, 0x00A4, 0x0490, 0x00A6, 0x00A7, /* 0xA0 */
      0x0401, 0x00A9, 0x0404, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x0407, /* 0xA0 */
      0x00B0, 0x00B1, 0x0406, 0x0456, 0x0491, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x0451, 0x2116, 0x0454, 0x00BB, 0x0458, 0x0405, 0x0455, 0x0457, /* 0xB0 */
      0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0xC0 */
      0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F, /* 0xC0 */
      0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0xD0 */
      0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 0x042F, /* 0xD0 */
      0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xE0 */
      0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F, /* 0xE0 */
      0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xF0 */
      0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_MS_1253: {
    if (c<0x80) break;
    static int const val[]= {
      0x20AC,      0, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0, 0x2030,      0, 0x2039,      0,      0,      0,      0, /* 0x80 */
      0, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0, 0x2122,      0, 0x203A,      0,      0,      0,      0, /* 0x90 */
      0x00A0, 0x0385, 0x0386, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9,      0, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x2015, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x0384, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x0388, 0x0389, 0x038A, 0x00BB, 0x038C, 0x00BD, 0x038E, 0x038F, /* 0xB0 */
      0x0390, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, /* 0xC0 */
      0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F, /* 0xC0 */
      0x03A0, 0x03A1,      0, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7, /* 0xD0 */
      0x03A8, 0x03A9, 0x03AA, 0x03AB, 0x03AC, 0x03AD, 0x03AE, 0x03AF, /* 0xD0 */
      0x03B0, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, /* 0xE0 */
      0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF, /* 0xE0 */
      0x03C0, 0x03C1, 0x03C2, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7, /* 0xF0 */
      0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03CC, 0x03CD, 0x03CE,      0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_MS_1254: {
    if (c<0x80) break;
    static int const val[]= {
      0x20AC,      0, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0x02C6, 0x2030, 0x0160, 0x2039, 0x0152,      0,      0,      0, /* 0x80 */
      0, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0x02DC, 0x2122, 0x0161, 0x203A, 0x0153,      0,      0, 0x0178, /* 0x90 */
      0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF, /* 0xB0 */
      0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7, /* 0xC0 */
      0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0x011E, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7, /* 0xD0 */
      0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x0130, 0x015E, 0x00DF, /* 0xD0 */
      0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0x011F, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7, /* 0xF0 */
      0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x0131, 0x015F, 0x00FF  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_MS_1255: {
    if (c<0x80) break;
    static int const val[]= {
      0x20AC,      0, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0x02C6, 0x2030,      0, 0x2039,      0,      0,      0,      0, /* 0x80 */
      0, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0x02DC, 0x2122,      0, 0x203A,      0,      0,      0,      0, /* 0x90 */
      0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x20AA, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x00D7, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x00B9, 0x00F7, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF, /* 0xB0 */
      0x05B0, 0x05B1, 0x05B2, 0x05B3, 0x05B4, 0x05B5, 0x05B6, 0x05B7, /* 0xC0 */
      0x05B8, 0x05B9,      0, 0x05BB, 0x05BC, 0x05BD, 0x05BE, 0x05BF, /* 0xC0 */
      0x05C0, 0x05C1, 0x05C2, 0x05C3, 0x05F0, 0x05F1, 0x05F2, 0x05F3, /* 0xD0 */
      0x05F4,      0,      0,      0,      0,      0,      0,      0, /* 0xD0 */
      0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5, 0x05D6, 0x05D7, /* 0xE0 */
      0x05D8, 0x05D9, 0x05DA, 0x05DB, 0x05DC, 0x05DD, 0x05DE, 0x05DF, /* 0xE0 */
      0x05E0, 0x05E1, 0x05E2, 0x05E3, 0x05E4, 0x05E5, 0x05E6, 0x05E7, /* 0xF0 */
      0x05E8, 0x05E9, 0x05EA,      0,      0, 0x200E, 0x200F,      0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_MS_1256: {
    if (c<0x80) break;
    static int const val[]= {
      0x20AC, 0x067E, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0x02C6, 0x2030, 0x0679, 0x2039, 0x0152, 0x0686, 0x0698, 0x0688, /* 0x80 */
      0x06AF, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0x06A9, 0x2122, 0x0691, 0x203A, 0x0153, 0x200C, 0x200D, 0x06BA, /* 0x90 */
      0x00A0, 0x060C, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x06BE, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x00B9, 0x061B, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x061F, /* 0xB0 */
      0x06C1, 0x0621, 0x0622, 0x0623, 0x0624, 0x0625, 0x0626, 0x0627, /* 0xC0 */
      0x0628, 0x0629, 0x062A, 0x062B, 0x062C, 0x062D, 0x062E, 0x062F, /* 0xC0 */
      0x0630, 0x0631, 0x0632, 0x0633, 0x0634, 0x0635, 0x0636, 0x00D7, /* 0xD0 */
      0x0637, 0x0638, 0x0639, 0x063A, 0x0640, 0x0641, 0x0642, 0x0643, /* 0xD0 */
      0x00E0, 0x0644, 0x00E2, 0x0645, 0x0646, 0x0647, 0x0648, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x0649, 0x064A, 0x00EE, 0x00EF, /* 0xE0 */
      0x064B, 0x064C, 0x064D, 0x064E, 0x00F4, 0x064F, 0x0650, 0x00F7, /* 0xF0 */
      0x0651, 0x00F9, 0x0652, 0x00FB, 0x00FC, 0x200E, 0x200F, 0x06D2  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_MS_1257: {
    if (c<0x80) break;
    static int const val[]= {
      0x20AC,      0, 0x201A,      0, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0, 0x2030,      0, 0x2039,      0, 0x00A8, 0x02C7, 0x00B8, /* 0x80 */
      0, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0, 0x2122,      0, 0x203A,      0, 0x00AF, 0x02DB,      0, /* 0x90 */
      0x00A0,      0, 0x00A2, 0x00A3, 0x00A4,      0, 0x00A6, 0x00A7, /* 0xA0 */
      0x00D8, 0x00A9, 0x0156, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00C6, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00F8, 0x00B9, 0x0157, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00E6, /* 0xB0 */
      0x0104, 0x012E, 0x0100, 0x0106, 0x00C4, 0x00C5, 0x0118, 0x0112, /* 0xC0 */
      0x010C, 0x00C9, 0x0179, 0x0116, 0x0122, 0x0136, 0x012A, 0x013B, /* 0xC0 */
      0x0160, 0x0143, 0x0145, 0x00D3, 0x014C, 0x00D5, 0x00D6, 0x00D7, /* 0xD0 */
      0x0172, 0x0141, 0x015A, 0x016A, 0x00DC, 0x017B, 0x017D, 0x00DF, /* 0xD0 */
      0x0105, 0x012F, 0x0101, 0x0107, 0x00E4, 0x00E5, 0x0119, 0x0113, /* 0xE0 */
      0x010D, 0x00E9, 0x017A, 0x0117, 0x0123, 0x0137, 0x012B, 0x013C, /* 0xE0 */
      0x0161, 0x0144, 0x0146, 0x00F3, 0x014D, 0x00F5, 0x00F6, 0x00F7, /* 0xF0 */
      0x0173, 0x0142, 0x015B, 0x016B, 0x00FC, 0x017C, 0x017E, 0x02D9  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_MS_1258: {
    if (c<0x80) break;
    static int const val[]= {
      0x20AC,      0, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, /* 0x80 */
      0x02C6, 0x2030,      0, 0x2039, 0x0152,      0,      0,      0, /* 0x80 */
      0, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, /* 0x90 */
      0x02DC, 0x2122,      0, 0x203A, 0x0153,      0,      0, 0x0178, /* 0x90 */
      0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, /* 0xA0 */
      0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF, /* 0xA0 */
      0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, /* 0xB0 */
      0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF, /* 0xB0 */
      0x00C0, 0x00C1, 0x00C2, 0x0102, 0x00C4, 0x00C5, 0x00C6, 0x00C7, /* 0xC0 */
      0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x0300, 0x00CD, 0x00CE, 0x00CF, /* 0xC0 */
      0x0110, 0x00D1, 0x0309, 0x00D3, 0x00D4, 0x01A0, 0x00D6, 0x00D7, /* 0xD0 */
      0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x01AF, 0x0303, 0x00DF, /* 0xD0 */
      0x00E0, 0x00E1, 0x00E2, 0x0103, 0x00E4, 0x00E5, 0x00E6, 0x00E7, /* 0xE0 */
      0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x0301, 0x00ED, 0x00EE, 0x00EF, /* 0xE0 */
      0x0111, 0x00F1, 0x0323, 0x00F3, 0x00F4, 0x01A1, 0x00F6, 0x00F7, /* 0xF0 */
      0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x01B0, 0x20AB, 0x00FF  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_CENTEURO: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C4, 0x0100, 0x0101, 0x00C9, 0x0104, 0x00D6, 0x00DC, 0x00E1, /* 0x80 */
      0x0105, 0x010C, 0x00E4, 0x010D, 0x0106, 0x0107, 0x00E9, 0x0179, /* 0x80 */
      0x017A, 0x010E, 0x00ED, 0x010F, 0x0112, 0x0113, 0x0116, 0x00F3, /* 0x90 */
      0x0117, 0x00F4, 0x00F6, 0x00F5, 0x00FA, 0x011A, 0x011B, 0x00FC, /* 0x90 */
      0x2020, 0x00B0, 0x0118, 0x00A3, 0x00A7, 0x2022, 0x00B6, 0x00DF, /* 0xA0 */
      0x00AE, 0x00A9, 0x2122, 0x0119, 0x00A8, 0x2260, 0x0123, 0x012E, /* 0xA0 */
      0x012F, 0x012A, 0x2264, 0x2265, 0x012B, 0x0136, 0x2202, 0x2211, /* 0xB0 */
      0x0142, 0x013B, 0x013C, 0x013D, 0x013E, 0x0139, 0x013A, 0x0145, /* 0xB0 */
      0x0146, 0x0143, 0x00AC, 0x221A, 0x0144, 0x0147, 0x2206, 0x00AB, /* 0xC0 */
      0x00BB, 0x2026, 0x00A0, 0x0148, 0x0150, 0x00D5, 0x0151, 0x014C, /* 0xC0 */
      0x2013, 0x2014, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x25CA, /* 0xD0 */
      0x014D, 0x0154, 0x0155, 0x0158, 0x2039, 0x203A, 0x0159, 0x0156, /* 0xD0 */
      0x0157, 0x0160, 0x201A, 0x201E, 0x0161, 0x015A, 0x015B, 0x00C1, /* 0xE0 */
      0x0164, 0x0165, 0x00CD, 0x017D, 0x017E, 0x016A, 0x00D3, 0x00D4, /* 0xE0 */
      0x016B, 0x016E, 0x00DA, 0x016F, 0x0170, 0x0171, 0x0172, 0x0173, /* 0xF0 */
      0x00DD, 0x00FD, 0x0137, 0x017B, 0x0141, 0x017C, 0x0122, 0x02C7  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_CROATIAN: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C4, 0x00C5, 0x00C7, 0x00C9, 0x00D1, 0x00D6, 0x00DC, 0x00E1, /* 0x80 */
      0x00E0, 0x00E2, 0x00E4, 0x00E3, 0x00E5, 0x00E7, 0x00E9, 0x00E8, /* 0x80 */
      0x00EA, 0x00EB, 0x00ED, 0x00EC, 0x00EE, 0x00EF, 0x00F1, 0x00F3, /* 0x90 */
      0x00F2, 0x00F4, 0x00F6, 0x00F5, 0x00FA, 0x00F9, 0x00FB, 0x00FC, /* 0x90 */
      0x2020, 0x00B0, 0x00A2, 0x00A3, 0x00A7, 0x2022, 0x00B6, 0x00DF, /* 0xA0 */
      0x00AE, 0x0160, 0x2122, 0x00B4, 0x00A8, 0x2260, 0x017D, 0x00D8, /* 0xA0 */
      0x221E, 0x00B1, 0x2264, 0x2265, 0x2206, 0x00B5, 0x2202, 0x2211, /* 0xB0 */
      0x220F, 0x0161, 0x222B, 0x00AA, 0x00BA, 0x03A9, 0x017E, 0x00F8, /* 0xB0 */
      0x00BF, 0x00A1, 0x00AC, 0x221A, 0x0192, 0x2248, 0x0106, 0x00AB, /* 0xC0 */
      0x010C, 0x2026, 0x00A0, 0x00C0, 0x00C3, 0x00D5, 0x0152, 0x0153, /* 0xC0 */
      0x0110, 0x2014, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x25CA, /* 0xD0 */
      0xF8FF, 0x00A9, 0x2044, 0x00A4, 0x2039, 0x203A, 0x00C6, 0x00BB, /* 0xD0 */
      0x2013, 0x00B7, 0x201A, 0x201E, 0x2030, 0x00C2, 0x0107, 0x00C1, /* 0xE0 */
      0x010D, 0x00C8, 0x00CD, 0x00CE, 0x00CF, 0x00CC, 0x00D3, 0x00D4, /* 0xE0 */
      0x0111, 0x00D2, 0x00DA, 0x00DB, 0x00D9, 0x0131, 0x02C6, 0x02DC, /* 0xF0 */
      0x00AF, 0x03C0, 0x00CB, 0x02DA, 0x00B8, 0x00CA, 0x00E6, 0x02C7  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_CYRILLIC: {
    if (c<0x80) break;
    static int const val[]= {
      0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0x80 */
      0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F, /* 0x80 */
      0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0x90 */
      0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 0x042F, /* 0x90 */
      0x2020, 0x00B0, 0x00A2, 0x00A3, 0x00A7, 0x2022, 0x00B6, 0x0406, /* 0xA0 */
      0x00AE, 0x00A9, 0x2122, 0x0402, 0x0452, 0x2260, 0x0403, 0x0453, /* 0xA0 */
      0x221E, 0x00B1, 0x2264, 0x2265, 0x0456, 0x00B5, 0x2202, 0x0408, /* 0xB0 */
      0x0404, 0x0454, 0x0407, 0x0457, 0x0409, 0x0459, 0x040A, 0x045A, /* 0xB0 */
      0x0458, 0x0405, 0x00AC, 0x221A, 0x0192, 0x2248, 0x2206, 0x00AB, /* 0xC0 */
      0x00BB, 0x2026, 0x00A0, 0x040B, 0x045B, 0x040C, 0x045C, 0x0455, /* 0xC0 */
      0x2013, 0x2014, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x201E, /* 0xD0 */
      0x040E, 0x045E, 0x040F, 0x045F, 0x2116, 0x0401, 0x0451, 0x044F, /* 0xD0 */
      0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xE0 */
      0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F, /* 0xE0 */
      0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xF0 */
      0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x00A4  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_GREEK: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C4, 0x00B9, 0x00B2, 0x00C9, 0x00B3, 0x00D6, 0x00DC, 0x0385, /* 0x80 */
      0x00E0, 0x00E2, 0x00E4, 0x0384, 0x00A8, 0x00E7, 0x00E9, 0x00E8, /* 0x80 */
      0x00EA, 0x00EB, 0x00A3, 0x2122, 0x00EE, 0x00EF, 0x2022, 0x00BD, /* 0x90 */
      0x2030, 0x00F4, 0x00F6, 0x00A6, 0x00AD, 0x00F9, 0x00FB, 0x00FC, /* 0x90 */
      0x2020, 0x0393, 0x0394, 0x0398, 0x039B, 0x039E, 0x03A0, 0x00DF, /* 0xA0 */
      0x00AE, 0x00A9, 0x03A3, 0x03AA, 0x00A7, 0x2260, 0x00B0, 0x00B7, /* 0xA0 */
      0x0391, 0x00B1, 0x2264, 0x2265, 0x00A5, 0x0392, 0x0395, 0x0396, /* 0xB0 */
      0x0397, 0x0399, 0x039A, 0x039C, 0x03A6, 0x03AB, 0x03A8, 0x03A9, /* 0xB0 */
      0x03AC, 0x039D, 0x00AC, 0x039F, 0x03A1, 0x2248, 0x03A4, 0x00AB, /* 0xC0 */
      0x00BB, 0x2026, 0x00A0, 0x03A5, 0x03A7, 0x0386, 0x0388, 0x0153, /* 0xC0 */
      0x2013, 0x2015, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x0389, /* 0xD0 */
      0x038A, 0x038C, 0x038E, 0x03AD, 0x03AE, 0x03AF, 0x03CC, 0x038F, /* 0xD0 */
      0x03CD, 0x03B1, 0x03B2, 0x03C8, 0x03B4, 0x03B5, 0x03C6, 0x03B3, /* 0xE0 */
      0x03B7, 0x03B9, 0x03BE, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BF, /* 0xE0 */
      0x03C0, 0x03CE, 0x03C1, 0x03C3, 0x03C4, 0x03B8, 0x03C9, 0x03C2, /* 0xF0 */
      0x03C7, 0x03C5, 0x03B6, 0x03CA, 0x03CB, 0x0390, 0x03B0, 0xF8A0  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_ICELAND: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C4, 0x00C5, 0x00C7, 0x00C9, 0x00D1, 0x00D6, 0x00DC, 0x00E1, /* 0x80 */
      0x00E0, 0x00E2, 0x00E4, 0x00E3, 0x00E5, 0x00E7, 0x00E9, 0x00E8, /* 0x80 */
      0x00EA, 0x00EB, 0x00ED, 0x00EC, 0x00EE, 0x00EF, 0x00F1, 0x00F3, /* 0x90 */
      0x00F2, 0x00F4, 0x00F6, 0x00F5, 0x00FA, 0x00F9, 0x00FB, 0x00FC, /* 0x90 */
      0x00DD, 0x00B0, 0x00A2, 0x00A3, 0x00A7, 0x2022, 0x00B6, 0x00DF, /* 0xA0 */
      0x00AE, 0x00A9, 0x2122, 0x00B4, 0x00A8, 0x2260, 0x00C6, 0x00D8, /* 0xA0 */
      0x221E, 0x00B1, 0x2264, 0x2265, 0x00A5, 0x00B5, 0x2202, 0x2211, /* 0xB0 */
      0x220F, 0x03C0, 0x222B, 0x00AA, 0x00BA, 0x03A9, 0x00E6, 0x00F8, /* 0xB0 */
      0x00BF, 0x00A1, 0x00AC, 0x221A, 0x0192, 0x2248, 0x2206, 0x00AB, /* 0xC0 */
      0x00BB, 0x2026, 0x00A0, 0x00C0, 0x00C3, 0x00D5, 0x0152, 0x0153, /* 0xC0 */
      0x2013, 0x2014, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x25CA, /* 0xD0 */
      0x00FF, 0x0178, 0x2044, 0x00A4, 0x00D0, 0x00F0, 0x00DE, 0x00FE, /* 0xD0 */
      0x00FD, 0x00B7, 0x201A, 0x201E, 0x2030, 0x00C2, 0x00CA, 0x00C1, /* 0xE0 */
      0x00CB, 0x00C8, 0x00CD, 0x00CE, 0x00CF, 0x00CC, 0x00D3, 0x00D4, /* 0xE0 */
      0xF8FF, 0x00D2, 0x00DA, 0x00DB, 0x00D9, 0x0131, 0x02C6, 0x02DC, /* 0xF0 */
      0x00AF, 0x02D8, 0x02D9, 0x02DA, 0x00B8, 0x02DD, 0x02DB, 0x02C7  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_ROMANIAN: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C4, 0x00C5, 0x00C7, 0x00C9, 0x00D1, 0x00D6, 0x00DC, 0x00E1, /* 0x80 */
      0x00E0, 0x00E2, 0x00E4, 0x00E3, 0x00E5, 0x00E7, 0x00E9, 0x00E8, /* 0x80 */
      0x00EA, 0x00EB, 0x00ED, 0x00EC, 0x00EE, 0x00EF, 0x00F1, 0x00F3, /* 0x90 */
      0x00F2, 0x00F4, 0x00F6, 0x00F5, 0x00FA, 0x00F9, 0x00FB, 0x00FC, /* 0x90 */
      0x2020, 0x00B0, 0x00A2, 0x00A3, 0x00A7, 0x2022, 0x00B6, 0x00DF, /* 0xA0 */
      0x00AE, 0x00A9, 0x2122, 0x00B4, 0x00A8, 0x2260, 0x0102, 0x015E, /* 0xA0 */
      0x221E, 0x00B1, 0x2264, 0x2265, 0x00A5, 0x00B5, 0x2202, 0x2211, /* 0xB0 */
      0x220F, 0x03C0, 0x222B, 0x00AA, 0x00BA, 0x03A9, 0x0103, 0x015F, /* 0xB0 */
      0x00BF, 0x00A1, 0x00AC, 0x221A, 0x0192, 0x2248, 0x2206, 0x00AB, /* 0xC0 */
      0x00BB, 0x2026, 0x00A0, 0x00C0, 0x00C3, 0x00D5, 0x0152, 0x0153, /* 0xC0 */
      0x2013, 0x2014, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x25CA, /* 0xD0 */
      0x00FF, 0x0178, 0x2044, 0x00A4, 0x2039, 0x203A, 0x0162, 0x0163, /* 0xD0 */
      0x2021, 0x00B7, 0x201A, 0x201E, 0x2030, 0x00C2, 0x00CA, 0x00C1, /* 0xE0 */
      0x00CB, 0x00C8, 0x00CD, 0x00CE, 0x00CF, 0x00CC, 0x00D3, 0x00D4, /* 0xE0 */
      0xF8FF, 0x00D2, 0x00DA, 0x00DB, 0x00D9, 0x0131, 0x02C6, 0x02DC, /* 0xF0 */
      0x00AF, 0x02D8, 0x02D9, 0x02DA, 0x00B8, 0x02DD, 0x02DB, 0x02C7  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_TURKISH: {
    if (c<0x80) break;
    static int const val[]= {
      0x00C4, 0x00C5, 0x00C7, 0x00C9, 0x00D1, 0x00D6, 0x00DC, 0x00E1, /* 0x80 */
      0x00E0, 0x00E2, 0x00E4, 0x00E3, 0x00E5, 0x00E7, 0x00E9, 0x00E8, /* 0x80 */
      0x00EA, 0x00EB, 0x00ED, 0x00EC, 0x00EE, 0x00EF, 0x00F1, 0x00F3, /* 0x90 */
      0x00F2, 0x00F4, 0x00F6, 0x00F5, 0x00FA, 0x00F9, 0x00FB, 0x00FC, /* 0x90 */
      0x2020, 0x00B0, 0x00A2, 0x00A3, 0x00A7, 0x2022, 0x00B6, 0x00DF, /* 0xA0 */
      0x00AE, 0x00A9, 0x2122, 0x00B4, 0x00A8, 0x2260, 0x00C6, 0x00D8, /* 0xA0 */
      0x221E, 0x00B1, 0x2264, 0x2265, 0x00A5, 0x00B5, 0x2202, 0x2211, /* 0xB0 */
      0x220F, 0x03C0, 0x222B, 0x00AA, 0x00BA, 0x03A9, 0x00E6, 0x00F8, /* 0xB0 */
      0x00BF, 0x00A1, 0x00AC, 0x221A, 0x0192, 0x2248, 0x2206, 0x00AB, /* 0xC0 */
      0x00BB, 0x2026, 0x00A0, 0x00C0, 0x00C3, 0x00D5, 0x0152, 0x0153, /* 0xC0 */
      0x2013, 0x2014, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x25CA, /* 0xD0 */
      0x00FF, 0x0178, 0x011E, 0x011F, 0x0130, 0x0131, 0x015E, 0x015F, /* 0xD0 */
      0x2021, 0x00B7, 0x201A, 0x201E, 0x2030, 0x00C2, 0x00CA, 0x00C1, /* 0xE0 */
      0x00CB, 0x00C8, 0x00CD, 0x00CE, 0x00CF, 0x00CC, 0x00D3, 0x00D4, /* 0xE0 */
      0xF8FF, 0x00D2, 0x00DA, 0x00DB, 0x00D9, 0xF8A0, 0x02C6, 0x02DC, /* 0xF0 */
      0x00AF, 0x02D8, 0x02D9, 0x02DA, 0x00B8, 0x02DD, 0x02DB, 0x02C7  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_APPLE_UKRAINIAN: {
    if (c<0x80) break;
    static int const val[]= {
      0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0x80 */
      0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F, /* 0x80 */
      0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0x90 */
      0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 0x042F, /* 0x90 */
      0x2020, 0x00B0, 0x0490, 0x00A3, 0x00A7, 0x2022, 0x00B6, 0x0406, /* 0xA0 */
      0x00AE, 0x00A9, 0x2122, 0x0402, 0x0452, 0x2260, 0x0403, 0x0453, /* 0xA0 */
      0x221E, 0x00B1, 0x2264, 0x2265, 0x0456, 0x00B5, 0x0491, 0x0408, /* 0xB0 */
      0x0404, 0x0454, 0x0407, 0x0457, 0x0409, 0x0459, 0x040A, 0x045A, /* 0xB0 */
      0x0458, 0x0405, 0x00AC, 0x221A, 0x0192, 0x2248, 0x2206, 0x00AB, /* 0xC0 */
      0x00BB, 0x2026, 0x00A0, 0x040B, 0x045B, 0x040C, 0x045C, 0x0455, /* 0xC0 */
      0x2013, 0x2014, 0x201C, 0x201D, 0x2018, 0x2019, 0x00F7, 0x201E, /* 0xD0 */
      0x040E, 0x045E, 0x040F, 0x045F, 0x2116, 0x0401, 0x0451, 0x044F, /* 0xD0 */
      0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xE0 */
      0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F, /* 0xE0 */
      0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xF0 */
      0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x00A4  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_KOI8_R: {
    if (c<0x80) break;
    static int const val[]= {
      0x2500, 0x2502, 0x250C, 0x2510, 0x2514, 0x2518, 0x251C, 0x2524, /* 0x80 */
      0x252C, 0x2534, 0x253C, 0x2580, 0x2584, 0x2588, 0x258C, 0x2590, /* 0x80 */
      0x2591, 0x2592, 0x2593, 0x2320, 0x25A0, 0x2219, 0x221A, 0x2248, /* 0x90 */
      0x2264, 0x2265, 0x00A0, 0x2321, 0x00B0, 0x00B2, 0x00B7, 0x00F7, /* 0x90 */
      0x2550, 0x2551, 0x2552, 0x0451, 0x2553, 0x2554, 0x2555, 0x2556, /* 0xA0 */
      0x2557, 0x2558, 0x2559, 0x255A, 0x255B, 0x255C, 0x255D, 0x255E, /* 0xA0 */
      0x255F, 0x2560, 0x2561, 0x0401, 0x2562, 0x2563, 0x2564, 0x2565, /* 0xB0 */
      0x2566, 0x2567, 0x2568, 0x2569, 0x256A, 0x256B, 0x256C, 0x00A9, /* 0xB0 */
      0x044E, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433, /* 0xC0 */
      0x0445, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, /* 0xC0 */
      0x043F, 0x044F, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432, /* 0xD0 */
      0x044C, 0x044B, 0x0437, 0x0448, 0x044D, 0x0449, 0x0447, 0x044A, /* 0xD0 */
      0x042E, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413, /* 0xE0 */
      0x0425, 0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, /* 0xE0 */
      0x041F, 0x042F, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412, /* 0xF0 */
      0x042C, 0x042B, 0x0417, 0x0428, 0x042D, 0x0429, 0x0427, 0x042A  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_UTF7: {
    // we must decode the complete string here
    --pos;
    while (pos < src.size()) {
      c=int(src[pos++]);
      if (c!=int('+')) {
        dest.push_back(uint32_t(c));
        continue;
      }
      bool firstWrite=false;
      int nBits=0;
      uint32_t actValue=0;
      while (pos<src.size()) {
        c=int(src[pos++]);
        if (c=='-') {
          if (!firstWrite) // +- is +
            dest.push_back(static_cast<uint32_t>('+'));
          else if (nBits>=6) {
            STOFF_DEBUG_MSG(("StarEncoding::read: arrgh nBits=%d\n", nBits));
          }
          break;
        }

        firstWrite=true;
        /* Index in Base64Tab or 0xFF, when is a invalid character */
        static uint8_t const charToId[128] = {
          0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     /* 0x00-0x07 */
          0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     /* 0x08-0x0F */
          0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     /* 0x10-0x17 */
          0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     /* 0x18-0x1F */
          0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     /* 0x20-0x27  !"#$%&' */
          0xFF, 0xFF, 0xFF,   62, 0xFF, 0xFF, 0xFF,   63,     /* 0x28-0x2F ()*+,-./ */
          52,   53,   54,   55,   56,   57,   58,   59,     /* 0x30-0x37 01234567 */
          60,   61, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     /* 0x38-0x3F 89:;<=>? */
          0xFF,    0,    1,    2,    3,    4,    5,    6,     /* 0x40-0x47 @ABCDEFG */
          7,    8,    9,   10,   11,   12,   13,   14,     /* 0x48-0x4F HIJKLMNO */
          15,   16,   17,   18,   19,   20,   21,   22,     /* 0x50-0x57 PQRSTUVW */
          23,   24,   25, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,     /* 0x58-0x5F XYZ[\]^_ */
          0xFF,   26,   27,   28,   29,   30,   31,   32,     /* 0x60-0x67 `abcdefg */
          33,   34,   35,   36,   37,   38,   39,   40,     /* 0x68-0x6F hijklmno */
          41,   42,   43,   44,   45,   46,   47,   48,     /* 0x70-0x77 pqrstuvw */
          49,   50,   51, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF      /* 0x78-0x7F xyz{|}~ */
        };
        if (c>=0x80 || charToId[c]==0xff) {
          STOFF_DEBUG_MSG(("StarEncoding::read: arrgh find for %x\n", static_cast<unsigned int>(c)));
          return false;
        }
        nBits+=6;
        actValue=(actValue<<6)|charToId[c];
        if (nBits<16) continue;
        unicode=uint32_t((actValue>>(nBits-16))&0xffff);
        actValue=actValue&((1<<(nBits-16))-1);
        dest.push_back(unicode);
        nBits-=16;
      }
    }
    return true;
  }
  case E_UTF8: { // create string directly
    bool bad=false;
    size_t numExtra=0;
    if ((c&0x80)==0x80) {
      if ((c&0x40)==0) bad=true;
      else if ((c&0x20)==0) numExtra=1;
      else if ((c&0x10)==0) numExtra=2;
      else if ((c&0x8)==0) numExtra=3;
      else if ((c&0x4)==0) numExtra=4;
      else if ((c&0x2)==0) numExtra=5;
      else bad=true;
    }
    if (bad||pos+numExtra>src.size()) {
      STOFF_DEBUG_MSG(("StarEncoding::read: can not read some caracter for %x\n", static_cast<unsigned int>(c)));
      return false;
    }
    dest.push_back(uint32_t(c));
    for (size_t i=0; i<numExtra; ++i) {
      c=char(src[pos++]);
      if ((c&0xc0)!=0x80) {
        STOFF_DEBUG_MSG(("StarEncoding::read: some extra caracter seems bad for %x, try to correct\n", static_cast<unsigned int>(c)));
        c=(c&0x3f)|0x80;
      }
      dest.push_back(static_cast<uint32_t>(c));
    }
    return true;
  }
  case E_KOI8_U: {
    if (c<0x80) break;
    static int const val[]= {
      0x2500, 0x2502, 0x250C, 0x2510, 0x2514, 0x2518, 0x251C, 0x2524, /* 0x80 */
      0x252C, 0x2534, 0x253C, 0x2580, 0x2584, 0x2588, 0x258C, 0x2590, /* 0x80 */
      0x2591, 0x2592, 0x2593, 0x2320, 0x25A0, 0x2219, 0x221A, 0x2248, /* 0x90 */
      0x2264, 0x2265, 0x00A0, 0x2321, 0x00B0, 0x00B2, 0x00B7, 0x00F7, /* 0x90 */
      0x2550, 0x2551, 0x2552, 0x0451, 0x0454, 0x2554, 0x0456, 0x0457, /* 0xA0 */
      0x2557, 0x2558, 0x2559, 0x255A, 0x255B, 0x0491, 0x255D, 0x255E, /* 0xA0 */
      0x255F, 0x2560, 0x2561, 0x0401, 0x0404, 0x2563, 0x0406, 0x0407, /* 0xB0 */
      0x2566, 0x2567, 0x2568, 0x2569, 0x256A, 0x0490, 0x256C, 0x00A9, /* 0xB0 */
      0x044E, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433, /* 0xC0 */
      0x0445, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, /* 0xC0 */
      0x043F, 0x044F, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432, /* 0xD0 */
      0x044C, 0x044B, 0x0437, 0x0448, 0x044D, 0x0449, 0x0447, 0x044A, /* 0xD0 */
      0x042E, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413, /* 0xE0 */
      0x0425, 0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, /* 0xE0 */
      0x041F, 0x042F, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412, /* 0xF0 */
      0x042C, 0x042B, 0x0417, 0x0428, 0x042D, 0x0429, 0x0427, 0x042A  /* 0xF0 */
    };
    unicode=static_cast<uint32_t>(val[c-0x80]);
    break;
  }
  case E_ISCII_DEVANAGARI: {
    if (c==0x9) {
      if (pos>=src.size()) return false;
      c=int(src[pos++]);
      if (c<1 || c>0x5a) {
        STOFF_DEBUG_MSG(("StarEncoding::read: find unexpected char 0x09%x\n", static_cast<unsigned int>(c)));
        break;
      }
      static int const val[]= {
        0x0901, 0x0902, 0x0903, 0x0905, 0x0906, 0x0907, 0x0908, /* A0 */
        0x0909, 0x090A, 0x090B, 0x090E, 0x090F, 0x0910, 0x090D, 0x0912,
        0x0913, 0x0914, 0x0911, 0x0915, 0x0916, 0x0917, 0x0918, 0x0919, /* B0 */
        0x091A, 0x091B, 0x091C, 0x091D, 0x091E, 0x091F, 0x0920, 0x0921,
        0x0922, 0x0923, 0x0924, 0x0925, 0x0926, 0x0927, 0x0928, 0x0929, /* C0 */
        0x092A, 0x092B, 0x092C, 0x092D, 0x092E, 0x092F, 0x095F, 0x0930,
        0x0931, 0x0932, 0x0933, 0x0934, 0x0935, 0x0936, 0x0937, 0x0938, /* D0 */
        0x0939,      0, 0x093E, 0x093F, 0x0940, 0x0941, 0x0942, 0x0943,
        0x0946, 0x0947, 0x0948, 0x0945, 0x094A, 0x094B, 0x094C, 0x0949, /* E0 */
        0x094D, 0x093C, 0x0964,      0,      0,      0,      0,      0,
        0, 0x0966, 0x0967, 0x0968, 0x0969, 0x096A, 0x096B, 0x096C, /* F0 */
        0x096D, 0x096E, 0x096F
      };
      unicode=static_cast<uint32_t>(val[c-1]);
      break;
    }
    if (c<0xa1 || c>0xfa) break;
    static int const val[]= {
      0x0901, 0x0902, 0x0903, 0x0905, 0x0906, 0x0907, 0x0908, /* A0 */
      0x0909, 0x090A, 0x090B, 0x090E, 0x090F, 0x0910, 0x090D, 0x0912,
      0x0913, 0x0914, 0x0911, 0x0915, 0x0916, 0x0917, 0x0918, 0x0919, /* B0 */
      0x091A, 0x091B, 0x091C, 0x091D, 0x091E, 0x091F, 0x0920, 0x0921,
      0x0922, 0x0923, 0x0924, 0x0925, 0x0926, 0x0927, 0x0928, 0x0929, /* C0 */
      0x092A, 0x092B, 0x092C, 0x092D, 0x092E, 0x092F, 0x095F, 0x0930,
      0x0931, 0x0932, 0x0933, 0x0934, 0x0935, 0x0936, 0x0937, 0x0938, /* D0 */
      0x0939,      0, 0x093E, 0x093F, 0x0940, 0x0941, 0x0942, 0x0943,
      0x0946, 0x0947, 0x0948, 0x0945, 0x094A, 0x094B, 0x094C, 0x0949, /* E0 */
      0x094D, 0x093C, 0x0964,      0,      0,      0,      0,      0,
      0, 0x0966, 0x0967, 0x0968, 0x0969, 0x096A, 0x096B, 0x096C, /* F0 */
      0x096D, 0x096E, 0x096F
    };
    unicode=static_cast<uint32_t>(val[c-0xa1]);
    break;
  }
  case E_JIS_X_0201: // ok
    if (c<0xa1 || c>=0xdf) break;
    unicode=0xfec0+uint32_t(c);
    break;
  case E_UCS4: // assume bigendian
    if (pos+3>=src.size()) return false;
    for (int i=0; i<3; ++i) unicode=uint32_t((unicode<<8)|src[pos++]);
    break;
  case E_UCS2: // assume bigendian
    if (pos+3>=src.size()) return false;
    unicode=uint32_t((unicode<<8)|src[pos++]);
    break;
  case E_SHIFT_JIS: // already done
  case E_MS_932: // already done
  case E_APPLE_CHINSIMP: // already done
  case E_MS_936: // already done
  case E_GB_2312: // already done
  case E_EUC_CN:  // already done
  case E_GBT_12345: // already done
  case E_GBK: // already done
  case E_APPLE_JAPANESE: // already done
  case E_JIS_X_0208: // already done
  case E_JIS_X_0212: // already done
  case E_EUC_JP: // already done
  case E_APPLE_CHINTRAD: // already done
  case E_MS_950: // already done
  case E_BIG5: // already done
  case E_APPLE_KOREAN: // already done
  case E_MS_949: // already done
  case E_EUC_KR: // already done
  case E_BIG5_HKSCS: // already done
  case E_MS_1361: // already done
  default:
    STOFF_DEBUG_MSG(("StarEncoding::read: unimplemented encoding %d\n", int(encoding)));
    break;
  }
  if (!unicode) {
    static int numError=0;
    if (++numError<10) {
      STOFF_DEBUG_MSG(("StarEncoding::read: unknown caracter %x\n", static_cast<unsigned int>(c)));
    }
  }
  dest.push_back(unicode);
  return true;
}
// vim: set filetype=cpp tabstop=2 shiftwidth=2 cindent autoindent smartindent noexpandtab:
