// Copyright (c) 2025 Valentin Lobstein (Chocapikk) <balgogan@protonmail.com>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package vulnerability

import (
	"strconv"
	"testing"
)

func BenchmarkGetVulnerabilitiesForPlugin_Small(b *testing.B) {
	vulnerabilities := []Vulnerability{
		{Slug: "plugin1", CVE: "CVE-2024-0001", FromVersion: "1.0.0", ToVersion: "2.0.0"},
		{Slug: "plugin1", CVE: "CVE-2024-0002", FromVersion: "2.0.0", ToVersion: "3.0.0"},
		{Slug: "plugin2", CVE: "CVE-2024-0003", FromVersion: "1.0.0", ToVersion: "2.0.0"},
		{Slug: "plugin3", CVE: "CVE-2024-0004", FromVersion: "1.0.0", ToVersion: "2.0.0"},
	}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		_ = GetVulnerabilitiesForPlugin(vulnerabilities, "plugin1", "1.5.0")
	}
}

func BenchmarkGetVulnerabilitiesForPlugin_Medium(b *testing.B) {
	vulnerabilities := make([]Vulnerability, 1000)
	for i := 0; i < 1000; i++ {
		vulnerabilities[i] = Vulnerability{
			Slug:        "plugin" + strconv.Itoa(i%100),
			CVE:         "CVE-2024-" + strconv.Itoa(i),
			FromVersion: "1.0.0",
			ToVersion:   "2.0.0",
		}
	}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		_ = GetVulnerabilitiesForPlugin(vulnerabilities, "plugin50", "1.5.0")
	}
}

func BenchmarkGetVulnerabilitiesForPlugin_Large(b *testing.B) {
	vulnerabilities := make([]Vulnerability, 10000)
	for i := 0; i < 10000; i++ {
		vulnerabilities[i] = Vulnerability{
			Slug:        "plugin" + strconv.Itoa(i%1000),
			CVE:         "CVE-2024-" + strconv.Itoa(i),
			FromVersion: "1.0.0",
			ToVersion:   "2.0.0",
		}
	}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		_ = GetVulnerabilitiesForPlugin(vulnerabilities, "plugin500", "1.5.0")
	}
}

func BenchmarkDetermineSeverityFromCVSS(b *testing.B) {
	scores := []float64{9.5, 7.5, 4.5, 2.0, 0.0}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		for _, score := range scores {
			_ = DetermineSeverityFromCVSS(score)
		}
	}
}

func BenchmarkDetermineSeverityFromTitle(b *testing.B) {
	titles := []string{
		"Critical SQL Injection Vulnerability",
		"High Privilege Escalation Issue",
		"Medium XSS Vulnerability",
		"Low Information Disclosure",
		"Regular Update",
	}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		for _, title := range titles {
			_ = DetermineSeverityFromTitle(title)
		}
	}
}

func BenchmarkDetermineAuthTypeFromCVSS(b *testing.B) {
	vectors := []string{
		"CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H",
		"CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H",
		"CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H",
		"CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L",
	}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		for _, vector := range vectors {
			_ = DetermineAuthTypeFromCVSS(vector)
		}
	}
}

